% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hybrid.R
\name{gena.hybrid}
\alias{gena.hybrid}
\title{Hybridization}
\usage{
gena.hybrid(
  population,
  fitness,
  hybrid.n = 1,
  method,
  par,
  opt.par,
  info = FALSE,
  iter = NULL,
  ...
)
}
\arguments{
\item{population}{numeric matrix which rows are chromosomes i.e. vectors of 
parameters values.}

\item{fitness}{numeric vector which \code{i}-th element is the value of 
\code{fn} at point \code{population[i, ]}.}

\item{hybrid.n}{positive integer representing the number of hybrids.}

\item{method}{hybridization method to improve chromosomes via local search.}

\item{par}{additional parameters to be passed depending on the \code{method}.}

\item{opt.par}{parameters of the local optimization function
to be used for hybridization algorithm (including \code{fn} and \code{gr}).}

\item{info}{logical; if \code{TRUE} then some optimization related 
information will be printed each iteration.}

\item{iter}{iteration number of the genetic algorithm.}

\item{...}{additional parameters to be passed to 
\code{fn} and \code{gr} functions.}
}
\value{
The function returns a list with the following elements:
\itemize{
\item \code{population} - matrix which rows are chromosomes including hybrids.
\item \code{fitness} - vector which i-th element is the fitness of the
i-th chromosome.
\item \code{hybrids.ind} - vector of indexes of chromosomes selected for
hybridization.
\item \code{counts} a two-element integer vector giving the number of
calls to \code{fn} and \code{gr} respectively.
}
}
\description{
Hybridization method (algorithm) to be used in the
genetic algorithm.
}
\details{
This function uses \code{\link[gena]{gena.mating}} function to 
select hybrids. Therefore \code{method} and \code{par} arguments will
be passed to this function. If some chromosomes selected to become hybrids
are duplicated then these duplicates will not be subject to local
optimization i.e. the number of hybrids will be decreased by the number
of duplicates (actual number of hybrids during some iterations may be 
lower than \code{hybrid.n}).

Currently \code{\link[stats]{optim}} is the only available local
optimizer. Therefore \code{opt.par} is a list containing parameters
that should be passed to \code{\link[stats]{optim}}.

For more information on hybridization
please see El-mihoub et. al. (2006).
}
\examples{
# Consider the following fitness function
fn <- function(x)
{
  val <- x[1] * x[2] - x[1] ^ 2 - x[2] ^ 2
}

# Also let's provide it's gradient (optional)
gr <- function(x)
{
  val <- c(x[2] - 2 * x[1],
           x[1] - 2 * x[2])
}

# Randomly initialize the population
set.seed(123)
n_population <- 10
population <- gena.population(pop.n = n_population,
                              lower = c(-5, -5), 
                              upper = c(5, 5))

# Calculate fitness of each chromosome
fitness <- rep(NA, n_population)
for(i in 1:n_population)
{
  fitness[i] <- fn(population[i, ])
}

# Perform hybridization
hybrids <- gena.hybrid(population = population,
                       fitness = fitness,
                       opt.par = list(fn = fn,
                                      gr = gr,
                                      method = "BFGS",
                                      control = list(fnscale = -1,
                                                     abstol = 1e-10,
                                                     reltol = 1e-10,
                                                     maxit = 1000)),
                       hybrid.n = 2,
                       method = "rank",
                       par = 0.8)
print(hybrids)

}
\references{
T. El-mihoub, A. Hopgood, L. Nolle, B. Alan (2006). 
Hybrid Genetic Algorithms: A Review.
\emph{Engineering Letters}, 13 (3), 124-137.
}
