qrs.fast.bt <- function(y, x, d, z, w0 = NULL, Q1, Q2, P = 10, link, family,
                        gridtheta, m, b0, reps, alpha) {
  #' qrs.fast.bt
  #'
  #' Algorithm 4: bootstrap algorithm with preprocessing and quantile grid 
  #' reduction for Quantile Regression with Selection (QRS).
  #'
  #' @param y = Dependent variable (N x 1)
  #' @param x = Regressors matrix (N x K)
  #' @param d = Participation variable (N x 1)
  #' @param z = Regressors and instruments matrix for the propensity score
  #' (N x Kz)
  #' @param w0 = Sample weights (N x 1)
  #' @param Q1 = Number of quantiles in reduced grid
  #' @param Q2 = Number of quantiles in large grid
  #' @param P =  Number of evaluated values of parameter with large quantile
  #' grid
  #' @param link = Link function to compute the propensity score
  #' @param family = Parametric copula family
  #' @param gridtheta = Grid of values for copula parameter (T x 1)
  #' @param m =  Parameter to select interval of observations in top and bottom
  #' groups
  #' @param b0 = Initial values of the beta coefficients for all quantiles in
  #' the reduced quantile grid (K x Q1)
  #' @param reps = Number of bootstrap repetitions
  #' @param alpha = Significance level
  #' 
  #' @return gammase = Bootstrapped standard error of gamma coefficients
  #' (Kz x 1)
  #' @return gammaub = Bootstrapped upper bound of confidence interval of gamma
  #' coefficients (Kz x 1)
  #' @return gammalb = Bootstrapped lower bound of confidence interval of gamma
  #' coefficients (Kz x 1)
  #' @return betase = Bootstrapped standard error of beta coefficients (K x Q)
  #' @return betaub = Bootstrapped upper bound of confidence interval of beta
  #' coefficients (K x Q)
  #' @return betalb = Bootstrapped lower bound of confidence interval of beta
  #' coefficients (K x Q)
  #' @return thetase = Bootstrapped standard error of theta coefficients
  #' (1 x 1)
  #' @return thetaub = Bootstrapped upper bound of confidence interval of theta
  #' coefficients (1 x 1)
  #' @return thetalb = Bootstrapped lower bound of confidence interval of theta
  #' coefficients (1 x 1)
  #' @return gamma = Bootstrapped estimated theta coefficients (Kz x reps)
  #' @return beta = Bootstrapped estimated beta coefficients (K x Q2 x reps)
  #' @return theta = Bootstrapped estimated copula parameter (1 x reps)
  #' @return objf = Bootstrapped value of objective function at the optimum
  #' (1 x reps)
  #'
  #' @export
  #' @importFrom stats binomial
  #' 
  #' @examples
  #' \donttest{
  #' set.seed(1)
  #' N <- 100
  #' x <- cbind(1, 2 + runif(N))
  #' z <- cbind(x, runif(N))
  #' cop <- copula::normalCopula(param = -0.5, dim = 2)
  #' copu <- copula::rCopula(N, cop)
  #' v <- copu[,1]
  #' u <- copu[,2]
  #' gamma <- c(-1.5, 0.05, 2)
  #' beta <- cbind(qnorm(u), u^0.5)
  #' prop <- exp(z %*% gamma) / (1 + exp(z %*% gamma))
  #' d <- as.numeric(v <= prop)
  #' y <- d * rowSums(x * beta)
  #' w <- matrix(1, nrow = N, ncol = 1)
  #' 
  #' Q1 <- 9
  #' Q2 <- 19
  #' P <- 2
  #' m <- 1
  #' gridtheta <- seq(-1, 0, by = 0.1)
  #' link <- "probit"
  #' family <- "Gaussian"
  #' reps <- 10
  #' alpha <- 0.05
  #' 
  #' est <- qrs.fast(y, x[,-1], d, z[,-1], w, Q1, Q2, P, link, family, gridtheta, m)
  #' bt <- qrs.fast.bt(y, x[,-1], d, z[,-1], w, Q1, Q2, P, link, family,
  #'                   gridtheta, m, est$b1, reps, alpha)
  #' summary(bt)
  #' }
  
  #  library(copula) # For copula functions
  
  x <- cbind(rep(1,NROW(y)), x)
  
  N <- NROW(x)
  K <- NCOL(x)
  Kz <- NCOL(z) + 1
  
  # Sample weights
  if (is.null(w0)) w0 <- rep(1, N)
  
  # Quantile grids
  gridq1 <- seq(1 / (Q1 + 1), Q1 / (Q1 + 1), length.out = Q1)
  gridq2 <- seq(1 / (Q2 + 1), Q2 / (Q2 + 1), length.out = Q2)
  
  # Prevent conditional copula values too close to 0 or 1
  eps <- 1e-5
  
  # Central value of theta parameter grid
  initt <- floor((length(gridtheta) + 1) / 2)
  
  # Initialize matrices to store results
  gamma <- array(0, dim = c(Kz, reps))
  b1 <- array(0, dim = c(K, Q1, length(gridtheta), reps))
  objf1 <- array(0, dim = c(length(gridtheta), reps))
  gridtheta2 <- array(0, dim = c(P, reps))
  b2 <- array(0, dim = c(K, Q2, P, reps))
  objf2 <- array(0, dim = c(P, reps))
  theta <- rep(0, reps)
  beta <- array(0, dim = c(K, Q2, reps))
  objf_min <- rep(0, reps)
  
  for (i0 in 1:reps) {
    # Bootstrap weights
    V <- stats::rexp(N, rate = 1)
    w <- w0 * V
    
    # Estimate the propensity score
    if (!link %in% c("probit", "logit")) {
      stop("Unsupported link function. Use 'probit' or 'logit'.")
    }
    ghat <- switch(link,
                   "probit" = stats::glm(d ~ z, family = binomial(link = "probit"), weights = w),
                   "logit" = stats::glm(d ~ z, family = binomial(link = "logit"), weights = w)
    )
    prop <- ghat$fitted.values
    gamma[,i0] <- ghat$coefficients
    
    # Weighted regressors
    xw <- x[d==1,] * matrix(w[d==1], nrow = sum(d), ncol = K, byrow = FALSE)
    N1 <- NROW(xw)
    
    # Conservative estimate of standard error
    small <- 1e-6
    zeta <- sqrt(rowSums((xw %*% solve(t(xw) %*% xw))^2))
    zeta <- pmax(zeta, small)
    
    
    # Instrument
    phi <- prop * w
    
    # Estimation with reduced quantile grid
    for (i1 in initt:length(gridtheta)) {
      t <- gridtheta[i1]
      
      # Create the copula based on the specified family
      copula <- switch(family,
                       "Gaussian" = copula::normalCopula(param = t, dim = 2),
                       "Clayton" = copula::claytonCopula(param = t, dim = 2),
                       "Frank" = copula::frankCopula(param = t, dim = 2),
                       "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                       stop("Unsupported copula family")
      )
      
      # Copula conditional on participation
      C <- copula::pCopula(cbind(rep(gridq1, times = N1), rep(prop[d==1], each = Q1)), copula)
      C <- matrix(C, nrow = N1, byrow = TRUE)
      G <- matrix(C / prop[d==1], nrow = N1, ncol = Q1)
      G <- pmin(pmax(G, eps), 1 - eps)
      
      # Slope parameters given copula
      if (i1 == initt) {
        b1[,,i1,i0] <- .rqrb0.fast(y[d==1], x[d==1,], w[d==1], G, zeta, m, b0[,,i1])
      } else {
        b1[,,i1,i0] <- .rqrb0.fast(y[d==1], x[d==1,], w[d==1], G, zeta, m, b1[,,i1 - 1, i0])
      }
      
      # Objective function for copula parameter
      objf1[i1, i0] <- ((t(phi[d==1]) %*% rowSums((y[d==1]<=x[d==1,]%*%b1[, , i1, i0]) - G)) / N1)^2
    }
    
    # Estimation for earlier gridtheta values
    for (i1 in (initt - 1):1) {
      t <- gridtheta[i1]
      
      # Create the copula based on the specified family
      copula <- switch(family,
                       "Gaussian" = copula::normalCopula(param = t, dim = 2),
                       "Clayton" = copula::claytonCopula(param = t, dim = 2),
                       "Frank" = copula::frankCopula(param = t, dim = 2),
                       "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                       stop("Unsupported copula family")
      )
      
      # Copula conditional on participation
      C <- copula::pCopula(cbind(rep(gridq1, times = N1), rep(prop[d==1], each = Q1)), copula)
      C <- matrix(C, nrow = N1, byrow = TRUE)
      G <- matrix(C / prop[d==1], nrow = N1, ncol = Q1)
      G <- pmin(pmax(G, eps), 1 - eps)
      
      # Slope parameters given copula
      b1[,,i1,i0] <- .rqrb0.fast(y[d==1], x[d==1,], w[d==1], G, zeta, m, b1[,,i1 + 1, i0])
      
      # Objective function for copula parameter
      objf1[i1, i0] <- ((t(phi[d==1]) %*% rowSums((y[d==1]<=x[d==1,]%*%b1[, , i1, i0]) - G)) / N1)^2
    }
    
    # Sort parameter values by objective function; select P candidate values
    idx <- order(objf1[,i0])
    gridtheta2[,i0] <- gridtheta[idx[1:P]]
    
    # Estimation with large quantile grid
    if (Q1 < Q2) {
      for (i1 in 1:P) {
        t <- gridtheta2[i1,i0]
        
        # Create the copula based on the specified family
        copula <- switch(family,
                         "Gaussian" = copula::normalCopula(param = t, dim = 2),
                         "Clayton" = copula::claytonCopula(param = t, dim = 2),
                         "Frank" = copula::frankCopula(param = t, dim = 2),
                         "Gumbel" = copula::gumbelCopula(param = t, dim = 2),
                         stop("Unsupported copula family")
        )
        
        # Copula conditional on participation
        C <- copula::pCopula(cbind(rep(gridq2, times = N1), rep(prop[d==1], each = Q2)), copula)
        C <- matrix(C, nrow = N1, byrow = TRUE)
        G <- matrix(C / prop[d==1], nrow = N1, ncol = Q2)
        G <- pmin(pmax(G, eps), 1 - eps)
        
        # Slope parameters for large quantile grid
        # Assign values already estimated
        for (i2 in 1:Q1) {
          initq2 <- which.min(abs(gridq1[i2] - gridq2))
          b2[,initq2,i1,i0] <- b1[,i2,idx[i1],i0]
        }
        
        # Estimate the rest using as preliminary estimate those with the same 
        # copula parameter and a close quantile
        for (i2 in (initq2):Q2) {
          if (all(b2[,i2,i1,i0] == 0)) {
            b2[,i2,i1,i0] <- .rqrtau.fast(y[d==1], x[d==1,], w[d==1], G[, i2], zeta, m, b2[,i2-1,i1,i0])
          }
        }
        for (i2 in (initq2 - 1):1) {
          if (all(b2[,i2,i1,i0] == 0)) {
            b2[,i2,i1,i0] <- .rqrtau.fast(y[d==1], x[d==1,], w[d==1], G[, i2], zeta, m, b2[,i2+1,i1,i0])
          }
        }
        
        # Objective function
        objf2[i1, i0] <- ((t(phi[d==1]) %*% rowSums((y[d==1]<=x[d==1,]%*%b2[, , i1, i0]) - G)) / N1)^2
      }
    } else {
      objf2[,i0] <- m[idx[1:P],i0]
      b2[,,,i0] <- b1[,,idx[1:P],i0]
    }
    
    # Find minimum
    if (P > 1) {
      argminf <- which.min(objf2[,i0])
    } else {
      argminf <- 1
    }
    
    theta[i0] <- gridtheta2[argminf,i0]
    beta[,,i0] <- b2[,,argminf,i0]
    objf_min[i0] <- objf2[argminf,i0]
  }
  
  # Standard deviations and confidence intervals
  Na <- length(alpha)
  
  gammam <- array(0, dim = Kz)
  gammase <- array(0, dim = Kz)
  gammaub <- array(0, dim = c(Kz, Na))
  gammalb <- array(0, dim = c(Kz, Na))
  for (i1 in 1:Kz) {
    gammabt <- .bt.results(gamma[i1,], alpha)
    gammam[i1] <- gammabt$m
    gammase[i1] <- gammabt$se
    gammaub[i1,] <- gammabt$ub
    gammalb[i1,] <- gammabt$lb
  }
  
  betam <- array(0, dim = c(K, Q2))
  betase <- array(0, dim = c(K, Q2))
  betaub <- array(0, dim = c(K, Q2, Na))
  betalb <- array(0, dim = c(K, Q2, Na))
  for (i1 in 1:K) {
    for (i2 in 1:Q2) {
      betabt <- .bt.results(beta[i1,i2,], alpha)
      betam[i1,i2] <- betabt$m
      betase[i1,i2] <- betabt$se
      betaub[i1,i2,] <- betabt$ub
      betalb[i1,i2,] <- betabt$lb
    }
  }
  
  thetabt <- .bt.results(theta, alpha)
  thetam <- thetabt$m
  thetase <- thetabt$se
  thetaub <- thetabt$ub
  thetalb <- thetabt$lb
  
  list(gammase = gammase, gammaub = gammaub, gammalb = gammalb,
       betase = betase, betaub = betaub, betalb = betalb,
       thetase = thetase, thetaub = thetaub, thetalb = thetalb,
       gamma = gamma, beta = beta, theta = theta, objf = objf_min)
}