% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tools.R
\name{MakePositiveDefinite}
\alias{MakePositiveDefinite}
\title{Making positive definite matrix}
\usage{
MakePositiveDefinite(
  omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = NULL,
  scale = TRUE,
  u_list = c(1e-10, 1),
  tol = .Machine$double.eps^0.25
)
}
\arguments{
\item{omega}{input matrix.}

\item{pd_strategy}{method to ensure that the generated precision matrix is
positive definite (and hence can be a covariance matrix). If
\code{pd_strategy="diagonally_dominant"}, the precision matrix is made
diagonally dominant by setting the diagonal entries to the sum of absolute
values on the corresponding row and a constant u. If
\code{pd_strategy="min_eigenvalue"}, diagonal entries are set to the sum of
the absolute value of the smallest eigenvalue of the precision matrix with
zeros on the diagonal and a constant u.}

\item{ev_xx}{expected proportion of explained variance by the first Principal
Component (PC1) of a Principal Component Analysis. This is the largest
eigenvalue of the correlation (if \code{scale_ev=TRUE}) or covariance (if
\code{scale_ev=FALSE}) matrix divided by the sum of eigenvalues. If
\code{ev_xx=NULL} (the default), the constant u is chosen by maximising the
contrast of the correlation matrix.}

\item{scale}{logical indicating if the proportion of explained variance by
PC1 should be computed from the correlation (\code{scale=TRUE}) or
covariance (\code{scale=FALSE}) matrix.}

\item{u_list}{vector with two numeric values defining the range of values to
explore for constant u.}

\item{tol}{accuracy for the search of parameter u as defined in
\code{\link[stats]{optimise}}.}
}
\value{
A list with: \item{omega}{positive definite matrix.} \item{u}{value
  of the constant u.}
}
\description{
Determines the diagonal entries of a symmetric matrix to make it is positive
definite.
}
\details{
Two strategies are implemented to ensure positive definiteness: by
  diagonally dominance or using eigendecomposition.

  A diagonally dominant symmetric matrix with positive diagonal entries is
  positive definite. With \code{pd_strategy="diagonally_dominant"}, the
  diagonal entries of the matrix are defined to be strictly higher than the
  sum of entries on the corresponding row in absolute value, which ensures
  diagonally dominance. Let \eqn{\Omega*} denote the input matrix with zeros
  on the diagonal and \eqn{\Omega} be the output positive definite matrix. We
  have:

  \eqn{\Omega_{ii} = \sum_{j = 1}^p | \Omega_{ij}* | + u}, where \eqn{u > 0}
  is a parameter.

  A matrix is positive definite if all its eigenvalues are positive. With
  \code{pd_strategy="diagonally_dominant"}, diagonal entries of the matrix
  are defined to be higher than the absolute value of the smallest eigenvalue
  of the same matrix with a diagonal of zeros. Let \eqn{\lambda_1} denote the
  smallest eigenvvalue of the input matrix \eqn{\Omega*} with a diagonal of
  zeros, and \eqn{v_1} be the corresponding eigenvector. Diagonal entries in
  the output matrix \eqn{\Omega} are defined as:

  \eqn{\Omega_{ii} = | \lambda_1 | + u}, where \eqn{u > 0} is a parameter.

  It can be showed that \eqn{\Omega} has stricly positive eigenvalues. Let
  \eqn{\lambda} and \eqn{v} denote any eigenpair of \eqn{\Omega*}:

  \eqn{\Omega* v = \lambda v}

  \eqn{\Omega* v + (| \lambda_1 | + u) v = \lambda v + (| \lambda_1 | + u) v}

  \eqn{(\Omega* + (| \lambda_1 | + u) I) v = (\lambda + | \lambda_1 | + u) v}

  \eqn{\Omega v = (\lambda + | \lambda_1 | + u) v}

  The eigenvalues of \eqn{\Omega} are equal to the eigenvalues of
  \eqn{\Omega*} plus \eqn{| \lambda_1 |}. The smallest eigenvalue of
  \eqn{\Omega} is \eqn{(\lambda_1 + | \lambda_1 | + u) > 0}.


  Considering the matrix to make positive definite is a precision matrix, its
  standardised inverse matrix is the correlation matrix. In both cases, the
  magnitude of correlations is controlled by the constant u.

  If \code{ev_xx=NULL}, the constant u is chosen to maximise the
  \code{\link{Contrast}} of the corresponding correlation matrix.

  If \code{ev_xx} is provided, the constant u is chosen to generate a
  correlation matrix with required proportion of explained variance by the
  first Principal Component, if possible. This proportion of explained
  variance is equal to the largest eigenvalue of the correlation matrix
  divided by the sum of its eigenvalues. If \code{scale=FALSE}, the
  covariance matrix is used instead of the correlation matrix for faster
  computations.
}
\examples{
# Simulation of a symmetric matrix
p <- 5
set.seed(1)
omega <- matrix(rnorm(p * p), ncol = p)
omega <- omega + t(omega)
diag(omega) <- 0

# Diagonal dominance maximising contrast
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant"
)
eigen(omega_pd$omega)$values # positive eigenvalues

# Diagonal dominance with specific proportion of explained variance by PC1
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = 0.55
)
lambda_inv <- eigen(cov2cor(solve(omega_pd$omega)))$values
max(lambda_inv) / sum(lambda_inv) # expected ev

# Version not scaled (using eigenvalues from the covariance)
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = 0.55, scale = FALSE
)
lambda_inv <- 1 / eigen(omega_pd$omega)$values
max(lambda_inv) / sum(lambda_inv) # expected ev

# Non-negative eigenvalues maximising contrast
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue"
)
eigen(omega_pd$omega)$values # positive eigenvalues

# Non-negative eigenvalues with specific proportion of explained variance by PC1
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue",
  ev_xx = 0.7
)
lambda_inv <- eigen(cov2cor(solve(omega_pd$omega)))$values
max(lambda_inv) / sum(lambda_inv)

# Version not scaled (using eigenvalues from the covariance)
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue",
  ev_xx = 0.7, scale = FALSE
)
lambda_inv <- 1 / eigen(omega_pd$omega)$values
max(lambda_inv) / sum(lambda_inv)
}
\references{
\insertRef{ourstabilityselection}{fake}
}
