% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitTsfm.R
\name{fitTsfm}
\alias{fitTsfm}
\alias{coef.tsfm}
\alias{fitted.tsfm}
\alias{residuals.tsfm}
\title{Fit a time series factor model using time series regression}
\usage{
fitTsfm(
  asset.names,
  factor.names,
  mkt.name = NULL,
  rf.name = NULL,
  data = data,
  fit.method = c("LS", "DLS", "Robust"),
  variable.selection = c("none", "stepwise", "subsets", "lars"),
  control = fitTsfm.control(),
  ...
)

\method{coef}{tsfm}(object, ...)

\method{fitted}{tsfm}(object, ...)

\method{residuals}{tsfm}(object, ...)
}
\arguments{
\item{asset.names}{vector of syntactically valid asset names, whose returns are the dependent 
variable in the factor model.}

\item{factor.names}{vector containing syntactically valid names of the factors.}

\item{mkt.name}{syntactically valid name of the column for market returns. Default is \code{NULL}.}

\item{rf.name}{syntactically valid name of the column for the risk free rate; if excess returns 
should be calculated for all assets and factors. Default is \code{NULL}.}

\item{data}{vector, matrix, data.frame, xts, timeSeries or zoo object
containing the columns \code{asset.names}, \code{factor.names}, and 
optionally, \code{mkt.name} and \code{rf.name}.}

\item{fit.method}{the estimation method, one of "LS", "DLS" or "Robust". 
See details. Default is "LS".}

\item{variable.selection}{the variable selection method, one of "none", 
"stepwise","subsets","lars". See details. Default is "none".}

\item{control}{list of control parameters. Refer to 
\code{\link{fitTsfm.control}} for details.}

\item{...}{arguments passed to \code{\link{fitTsfm.control}}}

\item{object}{a fit object of class \code{tsfm} which is returned by 
\code{fitTsfm}}
}
\value{
\code{fitTsfm} returns an object of class \code{"tsfm"} for which 
\code{print}, \code{plot}, \code{predict} and \code{summary} methods exist. 

The generic functions \code{coef}, \code{fitted} and  \code{residuals} 
extract various useful features of the fit object. 
Additionally, \code{fmCov} computes the covariance matrix for asset returns 
based on the fitted factor model.

An object of class \code{"tsfm"} is a list containing the following 
components:
\item{asset.fit}{list of fitted objects for each asset. Each object is of 
class \code{lm} if \code{fit.method="LS" or "DLS"}, class \code{lmrobdetMM} if 
the \code{fit.method="Robust"}, or class \code{lars} if 
\code{variable.selection="lars"}.}
\item{alpha}{N x 1 data.frame of estimated alphas.}
\item{beta}{N x K data.frame of estimated betas.}
\item{r2}{length-N vector of R-squared values.}
\item{resid.sd}{length-N vector of residual standard deviations.}
\item{fitted}{xts data object of fitted values; iff 
\code{variable.selection="lars"}}
\item{call}{the matched function call.}
\item{data}{xts data object containing the asset(s) and factor(s) returns.}
\item{asset.names}{syntactically valid asset.names as input.}
\item{factor.names}{syntactically valid factor.names as input.}
\item{mkt.name}{syntactically valid mkt.name as input}
\item{fit.method}{fit.method as input.}
\item{variable.selection}{variable.selection as input.}
Where N is the number of assets, K is the number of factors and T is the 
number of time periods.
}
\description{
Fits a time series (a.k.a. macroeconomic) factor model for one 
or more asset returns or excess returns using time series regression. 
Users can choose between ordinary least squares-LS, discounted least 
squares-DLS (or) robust regression. Several variable selection options  
including Stepwise, Subsets, Lars are available as well. An object of class 
\code{"tsfm"} is returned.
}
\details{
Typically, factor models are fit using excess returns. \code{rf.name} gives 
the option to supply a risk free rate variable to subtract from each asset 
return and factor to compute excess returns. 

Estimation method "LS" corresponds to ordinary least squares using 
\code{\link[stats]{lm}}, "DLS" is discounted least squares (weighted least 
squares with exponentially declining weights that sum to unity), and, 
"Robust" is robust regression (using \code{\link[RobStatTM]{lmrobdetMM}}). 

If \code{variable.selection="none"}, uses all the factors and performs no 
variable selection. Whereas, "stepwise" performs traditional stepwise 
LS or Robust regression (using \code{\link[stats]{step}} or 
\code{\link[RobStatTM]{step.lmrobdetMM}}), that starts from the initial set of 
factors and adds/subtracts factors only if the regression fit, as measured 
by the Bayesian Information Criterion (BIC) or Akaike Information Criterion 
(AIC), improves. And, "subsets" enables subsets selection using 
\code{\link[leaps]{regsubsets}}; chooses the best performing subset of any 
given size or within a range of subset sizes. Different methods such as 
exhaustive search (default), forward or backward stepwise, or sequential 
replacement can be employed. See \code{\link{fitTsfm.control}} for more 
details on the control arguments.
 
\code{variable.selection="lars"} corresponds to least angle regression 
using \code{\link[lars]{lars}} with variants "lasso" (default), "lar", 
"stepwise" or "forward.stagewise". Note: If \code{variable.selection="lars"}, 
\code{fit.method} will be ignored.

Argument \code{mkt.name} can be used to add market-timing factors to any of 
the above methods. Please refer to \code{\link{fitTsfmMT}}, a wrapper to 
\code{fitTsfm} for details.  

\subsection{Data Processing}{

Note about NAs: Before model fitting, incomplete cases are removed for 
every asset (return data combined with respective factors' return data) 
using \code{\link[stats]{na.omit}}. Otherwise, all observations in 
\code{data} are included.

Note about \code{asset.names} and \code{factor.names}: Spaces in column 
names of \code{data} will be converted to periods as \code{fitTsfm} works 
with \code{xts} objects internally and colnames won't be left as they are.
}
}
\examples{
 # load data
data(managers, package = 'PerformanceAnalytics')

fit <- fitTsfm(asset.names = colnames(managers[,(1:6)]),
               factor.names = colnames(managers[,(7:9)]), 
               data=managers)
summary(fit)
fitted(fit)

# example using "subsets" variable selection
fit.sub <- fitTsfm(asset.names=colnames(managers[,(1:6)]),
                   factor.names=colnames(managers[,(7:9)]), 
                   data=managers, 
                   variable.selection="subsets", 
                   method="exhaustive", 
                   nvmin=2) 

# example using "lars" variable selection and subtracting risk-free rate
fit.lar <- fitTsfm(asset.names=colnames(managers[,(1:6)]),
                   factor.names=colnames(managers[,(7:9)]), 
                   rf.name="US 3m TR", 
                   data=managers, 
                   variable.selection="lars", 
                   lars.criterion="cv") 
}
\references{
Christopherson, J. A., Carino, D. R., & Ferson, W. E. (2009). Portfolio 
performance measurement and benchmarking. McGraw Hill Professional.

Efron, B., Hastie, T., Johnstone, I., & Tibshirani, R. (2004). Least angle 
regression. The Annals of statistics, 32(2), 407-499. 

Hastie, T., Tibshirani, R., Friedman, J., Hastie, T., Friedman, J., & 
Tibshirani, R. (2009). The elements of statistical learning (Vol. 2, No. 1). 
New York: Springer.
}
\seealso{
The \code{tsfm} methods for generic functions: 
\code{\link{plot.tsfm}}, \code{\link{predict.tsfm}}, 
\code{\link{print.tsfm}} and \code{\link{summary.tsfm}}. 

And, the following extractor functions: \code{\link[stats]{coef}}, 
\code{\link[stats]{fitted}}, \code{\link[stats]{residuals}},
\code{\link{fmCov}}, \code{\link{fmSdDecomp}}, \code{\link{fmVaRDecomp}} 
and \code{\link{fmEsDecomp}}.

\code{\link{paFm}} for Performance Attribution.
}
\author{
Eric Zivot, Sangeetha Srinivasan and Yi-An Chen.
}
