#' CSEM Lord Method
#'
#' @description
#' Compute Lord's CSEM in classical test theory under the binomial model.
#'
#' @param ni A numeric value indicating the number of items (must be at least 2).
#'
#'
#' @return A list with:
#' \describe{
#'   \item{x}{Vector of raw scores from 0 to \code{ni}.}
#'   \item{csem}{Vector of Lord CSEM values corresponding to each raw score.}
#' }
#'
#' @examples
#' csem_lord(40)
#'
#' @export
csem_lord <- function(ni) {

  # basic checks ---------------------------------------------------------------
  if (missing(ni)) {
    stop("`ni` must be supplied as the number of items.")
  }
  if (!is.numeric(ni) || length(ni) != 1L) {
    stop("`ni` must be a single numeric value.")
  }
  if (ni < 2) {
    stop("`ni` must be at least 2 to compute CSEM.")
  }

  ni <- as.integer(ni)

  # raw-score support ---------------------------------------------------------
  r <- 0:ni

  # create vector to store CSEM -----------------------------------------------
  csemx <- numeric(ni + 1L)

  # loop over "true" raw scores i ---------------------------------------------
  for (i in 0:ni) {

    # take raw score as true score
    pi <- i / ni

    # binomial probabilities for each possible observed raw score r
    p <- stats::dbinom(r, size = ni, prob = pi)
    # same as: choose(ni, r) * pi^r * (1 - pi)^(ni - r)

    # expectations of R and R^2
    ex  <- sum(r * p)
    ex2 <- sum(r^2 * p)

    # Lord's CSEM
    seX <- sqrt(ni / (ni - 1)) * sqrt(ex2 - ex^2)

    # store CSEM for this true score (note i + 1 index)
    csemx[i + 1L] <- seX
  }

  # return raw scores and Lord CSEM -------------------------------------------
  return(list(x = r, csem = csemx))
}
