#' Summarizing Factor Copula Model Fits
#'
#' `Summary` method for objects of class `"fcm"`, returned by [fcm()].
#'
#' @param object An object of class `"fcm"`, typically output from [fcm()].
#' @param ... Additional arguments (ignored).
#'
#' @return Invisibly, a list with summary components:
#' \itemize{
#'   \item \code{grid}: the selected grid point
#'   \item \code{neighbors}: indices and coordinates of neighbors
#'   \item \code{coef}: parameter estimates with 95\% CI
#'   \item \code{objective}: negative log-likelihood
#'   \item \code{information criteria}: \code{c(AIC, BIC, AICc)}
#'   \item \code{args}: fitting arguments
#' }
#' @export
#'
#' @examples
#' \donttest{
#' data(fit)
#' summary(fit)
#' }
#'
#' @srrstats {G2.1} Ensure appropriate input class checking.
#' @srrstats {G2.10} Ensure robust subsetting and structured output.
summary.fcm <- function(object, ...) {
  stopifnot(inherits(object, "fcm"))

  grid <- object$grid
  neigh <- object$neigh
  coord <- object$coord

  cat("\n=== Summary of Factor Copula Model Fit ===\n")

  # Grid information
  if (!is.null(grid) && nrow(grid) >= 1) {
    cat(sprintf("Grid location: ID = %s | Longitude = %.3f | Latitude = %.3f\n",
                grid$ID[1], grid$lon[1], grid$lat[1]))
  }

  # Neighbors
  n_neigh <- length(neigh)
  cat(sprintf("\nNumber of neighbors: %d\n", n_neigh))

  if (n_neigh > 0) {
    cat("Neighbor coordinates:\n")
    for (i in seq_along(neigh[[1]])) {
      idx <- neigh[[1]][i]
      if (idx <= nrow(coord)) {
        cat(sprintf("  - ID = %d | Longitude = %.3f | Latitude = %.3f\n",
                    idx, coord[idx, 1], coord[idx, 2]))
      }
    }
  }

  # Model coefficients and criteria
  cat("\nModel Coefficients (95% CI):\n")
  if (is.null(object$boot)) {
    print(list("Estimate (Hessian)" = coef(object, method = "hessian", ci = 0.95)))
  } else {
    print(list("Estimate (Bootstrap)" = coef(object, method = "boot", ci = 0.95)))
  }

  # Likelihood and IC
  cat("\nObjective value (negative log-likelihood):\n")
  print(-logLik(object))

  cat("\nInformation criteria:\n")
  ic <- c(AIC(object), BIC(object), AICc(object))
  names(ic) <- c("AIC", "BIC", "AICc")
  print(ic)

  # Other fitting arguments
  cat("\nFitting arguments:\n")
  print(object$arg)

  cat("=== End of Summary ===\n")
}
