#'@title Clustering tuning (intrinsic metric)
#'@description Tune clustering hyperparameters by evaluating an intrinsic metric over a parameter grid and selecting the elbow (max curvature).
#'@param base_model base model for tuning
#'@param folds number of folds for cross-validation
#'@param ranges a list of hyperparameter ranges to explore
#'@return returns a `clu_tune` object.
#'@references
#' Satopaa, V. et al. (2011). Finding a “Kneedle” in a Haystack.
#'@examples
#'data(iris)
#'
#'# fit model
#'model <- clu_tune(cluster_kmeans(k = 0), ranges = list(k = 1:10))
#'
#'model <- fit(model, iris[,1:4])
#'model$k
#'@export
clu_tune <- function(base_model, folds=10, ranges=NULL) {
  obj <- dal_tune(base_model, folds, ranges)
  obj$base_model <- base_model
  obj$name <- ""
  class(obj) <- append("clu_tune", class(obj))
  return(obj)
}

#'@importFrom stats predict
#'@export
#'@exportS3Method fit clu_tune
fit.clu_tune <- function(obj, data, ...) {

  build_cluster <- function(obj, ranges, data) {
    model <- obj$base_model
    model <- set_params(model, ranges)
    result <- cluster(model, data)
    return(result)
  }

  prepare_ranges <- function(obj, ranges) {
    ranges <- expand.grid(ranges)
    ranges$key <- 1:nrow(ranges)
    obj$ranges <- ranges
    return(obj)
  }

  ranges <- obj$ranges

  obj <- prepare_ranges(obj, ranges)
  ranges <- obj$ranges
  ranges$metric <- NA

  n <- nrow(ranges)
  i <- 1
  if (n > 1) {
    msg <- rep("", n) # store errors per configuration
    for (i in 1:n) {
      err <- tryCatch(
        {
          clu <- build_cluster(obj, ranges[i,], data)
          ranges$metric[i] <- attr(clu, "metric")
          ""
        },
        error = function(cond) {
          err <- sprintf("tune: %s", as.character(cond))
        }
      )
      if (err != "") {
        msg[i] <- err
      }
    }
    # choose parameter at the elbow of metric curve (max curvature)
    myfit <- fit_curvature_max()
    res <- transform(myfit, ranges$metric)
    i <- res$x
  }
  model <- set_params(obj$base_model, ranges[i,])
  return(model)
}

