% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reconstruct_env.R
\name{reconstruct_env}
\alias{reconstruct_env}
\title{Reconstruct past environmental conditions using ecometric models}
\usage{
reconstruct_env(
  fossildata,
  model_out,
  inv_transform = NULL,
  ci = 0.05,
  match_nearest = TRUE,
  fossil_lon = NULL,
  fossil_lat = NULL,
  modern_id = NULL,
  modern_lon = NULL,
  modern_lat = NULL,
  crs_proj = 4326
)
}
\arguments{
\item{fossildata}{A data frame containing fossil trait summaries per fossil site.
Must include columns corresponding to the same two summary metrics used for modern communities,
using the column names specified by \code{fossil_summ_trait_1} and \code{fossil_summ_trait_2}.}

\item{model_out}{Output list from \code{run_ecometric_model()}, containing modern data, diagnostics, and model settings.}

\item{inv_transform}{A function to back-transform environmental estimates to the original scale.
Default is \code{exp(x) - 1}. If \code{NULL}, the inverse transform stored in \code{model_out} is used if available.}

\item{ci}{The width of the interval to calculate around the maximum likelihood estimate (default = 0.05).}

\item{match_nearest}{Logical; if TRUE, the function matches each fossil to its nearest modern point based on coordinates (default = TRUE).}

\item{fossil_lon}{Name of the longitude column in \code{fossildata}. Required if \code{match_nearest = TRUE}.}

\item{fossil_lat}{Name of the latitude column in \code{fossildata}. Required if \code{match_nearest = TRUE}.}

\item{modern_id}{Name of the unique ID column in modern points (e.g., "GlobalID").}

\item{modern_lon}{Name of the longitude column in modern points. Required if \code{match_nearest = TRUE}.}

\item{modern_lat}{Name of the latitude column in modern points. Required if \code{match_nearest = TRUE}.}

\item{crs_proj}{Coordinate reference system to use when converting fossil and modern data to sf format (default = EPSG:4326).}
}
\value{
A data frame (\code{fossildata}) with reconstructed environmental values and optional nearest modern point data. Includes the following additional columns:
\describe{
\item{fossil_bin_1}{Numeric bin index for the first trait axis (based on first summary metric of trait distribution of fossil communities).}
\item{fossil_bin_2}{Numeric bin index for the second trait axis (based on second summary metric of trait distribution of fossil communities).}
\item{fossil_env_est}{Maximum likelihood estimate of the environmental variable (on transformed scale if applicable).}
\item{fossil_minlimit}{Lower bound of the confidence interval around the environmental estimate (transformed scale).}
\item{fossil_maxlimit}{Upper bound of the confidence interval around the environmental estimate (transformed scale).}
\item{fossil_env_est_UN}{(Optional) Inverse-transformed environmental estimate, on the original scale.}
\item{fossil_minlimit_UN}{(Optional) Inverse-transformed lower bound of the confidence interval.}
\item{fossil_maxlimit_UN}{(Optional) Inverse-transformed upper bound of the confidence interval.}
\item{nearest_modern_point}{(Optional) ID of the nearest modern sampling point (if \code{match_nearest = TRUE}).}
\item{...}{Additional columns from the matched modern site if \code{match_nearest = TRUE} (e.g., observed environmental values).}
}
}
\description{
Uses fossil community trait summaries to reconstruct
past environmental conditions by projecting them onto a binned ecometric trait space
built from modern data. Optionally, it also assigns each fossil point to the nearest
modern sampling site to retrieve observed environmental data.
}
\examples{
\donttest{
# Load internal data
data("geoPoints", package = "commecometrics")
data("traits", package = "commecometrics")
data("spRanges", package = "commecometrics")
data("fossils", package = "commecometrics")

# Step 1: Summarize modern trait values at sampling points
traitsByPoint <- summarize_traits_by_point(
  points_df = geoPoints,
  trait_df = traits,
  species_polygons = spRanges,
  trait_column = "RBL",
  species_name_col = "sci_name",
  continent = FALSE,
  parallel = FALSE
)

# Step 2: Run an ecometric model with BIO12 (precipitation)
ecoModel <- ecometric_model(
  points_df = traitsByPoint$points,
  env_var = "precip",
  transform_fun = function(x) log(x + 1),
  inv_transform_fun = function(x) exp(x) - 1,
  min_species = 3
)

# Step 3: Reconstruct fossil environments
recon <- reconstruct_env(
  fossildata = fossils,
  model_out = ecoModel,
  match_nearest = TRUE,
  fossil_lon = "Long",
  fossil_lat = "Lat",
  modern_id = "ID",
  modern_lon = "Longitude",
  modern_lat = "Latitude"
)

}
}
