% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameter.R, R/parameter_prior.R
\name{parameter}
\alias{parameter}
\alias{par_expr}
\alias{par_const}
\alias{par_named}
\alias{par_range}
\alias{par_prior}
\title{Model Parameters}
\usage{
par_expr(expr)

par_const(constant)

par_named(name)

par_range(name, lower, upper)

par_prior(name, prior)
}
\arguments{
\item{expr}{An R expression.
This allows to define a parameter using an R expression.
It can contain other named parameters (e.g. \code{2 * a} will create an
parameter that is twice the value of an existing parameter \code{a}).
Make sure that the expression always evaluates
to a valid parameter value (a single numeric in almost all cases).}

\item{constant}{An R expression.
The constant value of the parameter.
Different to \code{expr}, the expression is evaluated immediately and
can not depend on other named parameters.}

\item{name}{Character. The name of the parameter. Must be unique in a model.}

\item{lower}{A numeric. The lower boundary of the parameter's range.}

\item{upper}{A numeric. The upper boundary of the parameter's range.}

\item{prior}{An expression. Evaluation this expression should give
a sample from the prior distribution you want for the parameter.
For example using \code{rnorm(1)} gives a standard normal prior.}
}
\description{
These functions add parameters to a model. Parameters can either
be used in features, or added directly to a model using the plus operator.
The value of parameters can be specified in the simulation command
(for \code{par_named} and \code{par_range}), sampled from a prior
distribution (\code{par_prior}) or can be derived from other parameters
(\code{par_expr}).
}
\section{Functions}{
\itemize{
\item \code{par_expr()}: Creates a parameter with value determined by evaluating an expression.

\item \code{par_const()}: Creates an parameter that is equal to a fixed value.
Different to \code{par_expr}, the value is evaluated on parameter creation.

\item \code{par_named()}: Creates an parameter whose value is specified via the
\code{pars} argument in \code{\link{simulate.coalmodel}}.

\item \code{par_range()}: Creates an parameter that can take a range of possible
values.
Similar to \code{\link{par_named}}, the value of the parameter
used in a simulation is set via the \code{pars} argument.
This is primarily intended for creating model parameters for
\pkg{jaatha}.

\item \code{par_prior()}: Creates a named parameter with a prior
distribution. Before each simulation, the expression for the prior
is evaluated. The resulting value can be used in
\code{\link{par_expr}} under the chosen name.

}}
\examples{
# A parameter (here for the mutation rate) that is always
# equal to '5':
model_base <- coal_model(20, 1) +
  sumstat_nucleotide_div()

model <- model_base +
  feat_mutation(par_const(5))
simulate(model)

# With using a prior:
model <- model_base +
  feat_mutation(par_prior("theta", rnorm(1, 5, .1)))
simulate(model)

# Using a named parater:
model <- model_base +
  feat_mutation(par_named("theta"))
simulate(model, pars = c(theta = 5))

# or similarly a ranged parameter:
model <- model_base +
  feat_mutation(par_range("theta", 1, 10))
simulate(model, pars = c(theta = 5))

# Expressions can be used to derive parameters from
# other parameters:
model <- model_base +
  par_named("theta_half") +
  feat_mutation(par_expr(theta_half * 2))
simulate(model, pars = c(theta_half = 2.5))

model <- model_base +
  par_named("theta_log") +
  feat_mutation(par_expr(exp(theta_log)))
simulate(model, pars = c(theta_log = log(5)))
}
\seealso{
For parameters that variate between the loci in a model:
\code{\link{par_variation}}, \code{\link{par_zero_inflation}}
}
\author{
Paul Staab
}
