% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curve-fit.R
\name{average_curve_optim}
\alias{average_curve_optim}
\title{Generate an average curve using \code{optim}}
\usage{
average_curve_optim(
  data,
  coupon_var,
  x_var,
  y_var,
  fn,
  par,
  n_bins = 100,
  method = "L-BFGS-B",
  ...
)
}
\arguments{
\item{data}{a \code{data.frame}}

\item{coupon_var}{the variable for coupon identification}

\item{x_var}{the independent variable}

\item{y_var}{the dependent variable}

\item{fn}{a function defining the relationship between \code{Y} and \code{X}.
See Details for more information.}

\item{par}{the initial guess for the parameters}

\item{n_bins}{the number of bins to average the data inside into before
fitting}

\item{method}{The method to be used by \code{optim()}. Defaults to "L-BFGS-B"}

\item{...}{extra parameters to be passed to \code{optim()}}
}
\value{
an object of class \code{average_curve_optim} with the following content:
\itemize{
\item \code{data} the original data provided to the function
\item \code{binned_data} the data after the binning/averaging operation
\item \code{fn} the function supplied
\item \code{fit_optim} the results of the call to \code{optim}
\item \code{call} the call
\item \code{n_bins} the number of bins specified by the user
\item \code{max_x} the upper end of the range used for fitting
\item \code{y_var} the independent (\code{Y}) variable
\item \code{x_var} the dependent (\code{X}) variable
}
}
\description{
The user must decide on a single dependent variable (\code{Y}) and a
single independent variable (\code{X}). The user will specify a function defining
the relationship between the dependent and independent variables.
For a \code{data.frame} containing stress-strain (or load-deflection) data for
more than one coupon, the maximum value of \code{X} for each coupon is found and
the smallest maximum value determines the range over which the curve
fit is performed: the range is from zero to this value. Only positive
values of \code{X} are considered. For each coupon individually, the data is
divided into a user-specified number of bins and averaged within each bin.
The resulting binned/averaged data is then used for curve fitting.
The mean squared error between the observed value of \code{Y} and the result of
the user-specified function evaluated at each \code{X} is minimized by varying
the parameters \code{par}.
}
\details{
The function \code{fn} must have two arguments. The first argument must be the
value of the independent variable (\code{X}): this must be a numeric value
(of length one). The second argument must be a vector of the parameters of
the model, which are to be varied in order to obtain the best fit. See below
for an example.
}
\examples{
# using the `pa12_tension` dataset and fitting a cubic polynomial with
# zero intercept:
curve_fit <- average_curve_optim(
  pa12_tension,
  Coupon,
  Strain,
  Stress,
  function(strain, par) {
    sum(par * c(strain, strain^2, strain^3))
  },
  c(c1 = 1, c2 = 1, c3 = 1),
  n_bins = 100
)
## Range: ` Strain ` in  [ 0,  0.1409409 ]
##
## Call:
## average_curve_optim(data = pa12_tension, coupon_var = Coupon,
##                     x_var = Strain, y_var = Stress,
##                     fn = function(strain, par) {
##                       sum(par * c(strain, strain^2, strain^3))
##                     }, par = c(c1 = 1, c2 = 1, c3 = 1), n_bins = 100)
##
## Parameters:
##       c1        c2        c3
## 1174.372 -8783.106 20585.898

}
\seealso{
\code{\link[stats:optim]{optim()}}, \code{\link[=average_curve_lm]{average_curve_lm()}},
\code{\link[=print.average_curve_optim]{print.average_curve_optim()}}, \code{\link[=augment.average_curve_optim]{augment.average_curve_optim()}}
}
