% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simFCGDINA.R
\name{simFCGDINA}
\alias{simFCGDINA}
\title{Forced-choice data simulation based on the G-DINA model}
\usage{
simFCGDINA(
  N,
  Q.items,
  n.blocks = NULL,
  polarity = NULL,
  att = NULL,
  model = "GDINA",
  GDINA.args = list(GS = NULL, GS.items = c(1/3, 1/3), AC = 0, AT = 0),
  FCDCM.args = list(d0 = c(0.2, 0.2), sd = c(0.15, 0.15), a = c(0, 0), b = 0),
  seed = NULL
)
}
\arguments{
\item{N}{A \code{numeric} value indicating the sample size.}

\item{Q.items}{A binary \code{matrix} of dimensions \code{J} statements x \code{K} attributes indicating what statements measure what attributes.}

\item{n.blocks}{A \code{numeric} value indicating the number of forced-choice blocks.}

\item{polarity}{A \code{matrix} of dimensions \code{n.blocks} x 2 indicating whether each statament in each block is direct (1) or inverse (-1). Default is a matrix full of 1 (i.e., all statements are direct).}

\item{att}{A \code{matrix} of dimensions \code{N} individuals x \code{K} attributes indicating the attribute profile of each individual. Default is \code{NULL}, meaning that attribute profiles will be simulated based on the specifications listed on \code{GDINA.args} or \code{FCDCM.args}.}

\item{model}{Use the G-DINA model (\code{"GDINA"}) or the FC-DCM (\code{"FCDCM"}) as the generating model. Default is \code{"GDINA"}.}

\item{GDINA.args}{A \code{list} of arguments used if \code{model = "GDINA"}.
\describe{
\item{\code{GS}}{A \code{J} statements x 2 matrix indicating the guessing and slip parameter of each statement. Default is \code{NULL}.}
\item{\code{GS.items}}{Only used if \code{GDINA.args$GS = NULL}. A \code{vector} of length 2 indicating the minimum and maximum value for the random generation of guessing and slip parameters for each statement. Default is \code{c(1/3, 1/3).}}
\item{\code{AC}}{A \code{numeric} value indicating the attribute correlations in line with the multivariate normal threshold model (Chiu et al., 2009). Default is 0.}
\item{\code{AT}}{A \code{numeric} value indicating the attribute thresholds in line with the multivariate normal threshold model (Chiu et al., 2009). Default is 0.}
}}

\item{FCDCM.args}{A \code{list} of arguments used if \code{model = "FCDCM"}.
\describe{
\item{\code{d0}}{A \code{vector} of length 2 indicating the minimum and maximum value for the baseline probability for each FC block (see Huang, 2023). Default is \code{c(0.2, 0.2).}}
\item{\code{sd}}{A \code{vector} of length 2 indicating the minimum and maximum value for the statement utility parameters (see Huang, 2023). Default is \code{c(0.15, 0.15).}}
\item{\code{a}}{A \code{numeric} value indicating the minimum and maximum discrimination parameter for the higher-order model. Default is \code{c(0, 0).}}
\item{\code{b}}{A \code{numeric} value indicating the location parameter for the higher-order model. Default is 0.}
}}

\item{seed}{Random number generation seed. Default is \code{NULL}.}
}
\value{
\code{simFCGDINA} returns an object of class \code{simFCGDINA}.
\describe{
\item{\code{dat}}{Generated FC responses (\code{matrix}).}
\item{\code{att}}{Generated attribute profiles (\code{matrix}).}
\item{\code{Q}}{Generated Q-matrix of FC blocks (\code{matrix}).}
\item{\code{LCprob}}{Generated block response probabilities for each latent class (\code{matrix}).}
\item{\code{item.pairs}}{Statements used in each FC block (\code{matrix}).}
\item{\code{q_att}}{Attribute measured by each statement as used by Huang (2023) (\code{matrix}).}
\item{\code{q_sta}}{Relative position of each statement as used by Huang (2023) (\code{matrix}).}
\item{\code{simGDINA}}{Object of class \code{simGDINA} (\code{list}).}
\item{\code{polarity}}{Polarity matrix indicating the direction of each statement in each block (\code{matrix}).}
\item{\code{GS}}{Generated guessing and slip parameter for each statement (\code{matrix}).}
}
}
\description{
Simulate forced-choice (FC) responses based on the G-DINA model (de la Torre, 2011) and the FC-DCM (Huang, 2023).
This function accommodates FC responses to the \code{simGDINA} function from the \code{GDINA} package (Ma & de la Torre, 2020).
}
\examples{
\donttest{
library(GDINA)
set.seed(123)
# Q-matrix for the unidimensional statements
Q.items <- do.call("rbind", replicate(5, diag(5), simplify = FALSE))
# Guessing and slip
GS <- cbind(runif(n = nrow(Q.items), min = 0.1, max = 0.3),
            runif(n = nrow(Q.items), min = 0.1, max = 0.3))
n.blocks <- 30 # Number of forced-choice blocks
# Block polarity (1 = direct statement; -1 = indirect statement)
polarity <- matrix(1, nrow = n.blocks, ncol = 2)
sim <- simFCGDINA(N = 1000, Q.items, n.blocks = n.blocks, polarity = polarity,
                  model = "GDINA", GDINA.args = list(GS = GS), seed = 123)
}
}
\references{
Huang, H.-Y. (2023). Diagnostic Classification Model for Forced-Choice Items and Noncognitive Tests. \emph{Educational and Psychological Measurement}, \emph{83}(1), 146-180. https://doi.org/10.1177/00131644211069906

Ma, W., & de la Torre, J. (2020). GDINA: An R package for cognitive diagnosis modeling. \emph{Journal of Statistical Software}, \emph{93}(14). https://doi.org/10.18637/jss.v093.i14
}
\author{
{Pablo Nájera, Universidad Pontificia Comillas}
}
