% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RDINA.R
\name{RDINA}
\alias{RDINA}
\title{Restricted DINA model}
\usage{
RDINA(
  dat,
  Q,
  gate = "AND",
  att.prior = NULL,
  est = "Brent",
  tau.alpha = "MAP",
  emp.bayes = FALSE,
  boot = FALSE,
  n.boots = 500,
  n.cores = 1,
  maxitr = 1000,
  conv.crit = 1e-04,
  init.phi = 0.2,
  bound.p = 1e-06,
  verbose = TRUE,
  seed = NULL
)
}
\arguments{
\item{dat}{A \emph{N} individuals x \emph{J} items (\code{matrix} or \code{data.frame}). Missing values need to be coded as \code{NA}. Caution is advised if missing data are present.}

\item{Q}{A \emph{J} items x \emph{K} attributes Q-matrix (\code{matrix} or \code{data.frame}).}

\item{gate}{Either a conjunctive (\code{"AND"}) or disjunctive (\code{"OR"}) condensation rule to estimate the RDINA or RDINO model, respectively. Default is \code{"AND"}.}

\item{att.prior}{A 2^\emph{K} attributes vector containing the prior distribution for each latent class. The sum of all elements does not have to be equal to 1, since the vector will be normalized. Default is \code{NULL}, which is a uniform prior distribution.}

\item{est}{Use the Brent's method (\code{"Brent"}) or the expectation-maximization algorithm (\code{"EM"}) to estimate the model? Default is \code{"Brent"}, since it is faster and both algorithms are virtually equivalent for the RDINA/RDINO model.}

\item{tau.alpha}{Attribute profile estimator (either \code{"MAP"}, \code{"EAP"}, or \code{"MLE"}) used to calculate the estimated classification accuracy as done with the \code{CA} function of the \code{GDINA} package (Ma & de la Torre, 2020).}

\item{emp.bayes}{Use empirical Bayes estimation for structural parameters. Default is \code{FALSE}.}

\item{boot}{Use bootstrapping to increase robustness in posterior probabilities estimation. Default is \code{FALSE}.}

\item{n.boots}{Number of bootstrapping samples. Default is 500.}

\item{n.cores}{Number of CPU processors to speed up computation when bootstrapping is used. Default is 1.}

\item{maxitr}{Maximum number of iterations. Default is 1000.}

\item{conv.crit}{Convergence criterion regarding the maximum absolute change in either the phi parameter estimate or the marginal posterior probabilities of attribute mastery. Default is 0.0001.}

\item{init.phi}{Initial value for the phi parameter. Default is 0.2.}

\item{bound.p}{Lowest value for probability estimates (highest would be 1 - bound.p). Default is 1e-06.}

\item{verbose}{Print information after each iteration. Default is \code{TRUE}.}

\item{seed}{Random number generation seed (e.g., to solve ties in case they occur with MLE or MAP estimation). Default is \code{NULL}, which means that no specific seed is used.}
}
\value{
\code{RDINA} returns an object of class \code{RDINA}.
\describe{
\item{\code{MLE}}{Estimated attribute profiles with the MLE estimator (\code{matrix}).}
\item{\code{MAP}}{Estimated attribute profiles with the MAP estimator (\code{matrix}).}
\item{\code{EAP}}{Estimated attribute profiles with the EAP estimator (\code{matrix}).}
\item{\code{phi}}{Phi parameter estimate (\code{numeric}).}
\item{\code{post.probs}}{A (\code{list}) containing the estimates of the posterior probability of each examinee in each latent class (\code{pp}), marginal posterior probabilities of attribute mastery (\code{mp}), and posterior probability of each latent class (\code{lp}).}
\item{\code{likelihood}}{A (\code{list}) containing the likelihood of each examinee in each latent class (\code{lik_il}) and the model log-likelihood (\code{logLik}).}
\item{\code{test.fit}}{Relative model fit indices (\code{list}).}
\item{\code{class.accu}}{A (\code{list}) containing the classification accuracy estimates at the test-level (\code{tau}), latent class-level (\code{tau_l}), and attribute-level (\code{tau_k}).}
\item{\code{specifications}}{Function call specifications (\code{list}).}
}
}
\description{
Estimation of the \emph{restricted deterministic input, noisy "and" gate} model (R-DINA; Nájera et al., 2023).
In addition to the non-compensatory (i.e., conjunctive) condensation rule of the DINA model, the compensatory (i.e., disjunctive) rule of the DINO model can be also applied (i.e., R-DINO model).
The R-DINA/R-DINO model should be only considered for applications involving very small sample sizes (N < 100; Nájera et al., 2023), and model fit evaluation and comparison with competing models (e.g., DINA/DINO, G-DINA) is highly recommended.
}
\examples{
\donttest{
library(GDINA)
Q <- sim30GDINA$simQ # Q-matrix
K <- ncol(Q)
J <- nrow(Q)
set.seed(123)
GS <- data.frame(guessing = rep(0.2, J), slip = rep(0.2, J))
sim <- simGDINA(20, Q, GS, model = "DINA")
simdat <- sim$dat # Simulated data
simatt <- sim$attribute # Generating attributes
fit.RDINA <- RDINA(simdat, Q) # Apply the GNPC method
ClassRate(fit.RDINA$EAP, simatt) # Check classification accuracy
}
}
\references{
Ma, W., & de la Torre, J. (2020). GDINA: An R package for cognitive diagnosis modeling. \emph{Journal of Statistical Software}, \emph{93}(14). https://doi.org/10.18637/jss.v093.i14

Nájera, P., Abad, F. J., Chiu, C.-Y., & Sorrel, M. A. (2023). The Restricted DINA model: A Comprehensive Cognitive Diagnostic Model for Classroom-Level Assessments. \emph{Journal of Educational and Behavioral Statistics}.
}
\author{
{Pablo Nájera, Universidad Pontificia Comillas}
}
