% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make-tikz.R
\name{make_tikz}
\alias{make_tikz}
\title{Generate TikZ Code from a Causal Graph}
\usage{
make_tikz(
  x,
  ...,
  scale = 10,
  full_doc = TRUE,
  bend_edges = FALSE,
  bend_angle = 25,
  tier_label_pos = c("above", "below", "left", "right")
)
}
\arguments{
\item{x}{A \code{Disco}, \code{Knowledge}, or \code{caugi::caugi} object.}

\item{...}{Additional arguments passed to \code{\link[=plot]{plot()}} and \code{\link[caugi:plot]{caugi::plot()}}.}

\item{scale}{Numeric scalar. Scaling factor for node coordinates. Default is \code{10}.}

\item{full_doc}{Logical. If \code{TRUE} (default), generates a full standalone
LaTeX document. If \code{FALSE}, returns only the \code{tikzpicture} environment.}

\item{bend_edges}{Logical. If \code{TRUE}, edges are drawn with bent edges. Default is \code{FALSE}.
Edges connecting the same pair of nodes in both directions (\code{A \%-->\% B} and \code{B \%-->\% A})
are automatically bent left and right to avoid overlap. Bend direction is automatically chosen
to reduce overlap.}

\item{bend_angle}{Numeric scalar. Angle in degrees for bending arrows when
\code{bend_edges = TRUE}. Default is \code{25}.}

\item{tier_label_pos}{Character string specifying the position of tier labels
relative to the tier rectangles. Must be one of \code{"above"}, \code{"below"}, \code{"left"}, or \code{"right"}.
Default is \code{"above"}.}
}
\value{
A character string containing LaTeX TikZ code. Depending on
\code{full_doc}, this is either:
\itemize{
\item a complete LaTeX document (\code{full_doc = TRUE}), or
\item only the \code{tikzpicture} environment (\code{full_doc = FALSE}).
}
}
\description{
Generates LaTeX TikZ code from a \code{Disco}, \code{Knowledge}, or
\code{caugi::caugi} object, preserving node positions, labels, and visual styles.
Edges are rendered with arrows, line widths, and colors.
The output is readable LaTeX code that can be
directly compiled or modified.
}
\details{
The function calls \code{plot()} to generate a \code{caugi::caugi_plot} object, then
traverses the plot object's grob structure to extract nodes and
edges. Supported features include:
\itemize{
\item \strong{Nodes}
\itemize{
\item Fill color and draw color (supports both named colors and custom RGB values)
\item Font size
\item Coordinates are scaled by the \code{scale} parameter
}
\item \strong{Edges}
\itemize{
\item Line color and width
\item Arrow scale
\item Optional bending to reduce overlapping arrows
}
}

The generated TikZ code uses global style settings, and edges are connected
to nodes by name (as opposed to hard-coded coordinates), making it easy to
modify the output further if needed.
}
\examples{
################# Convert Knowledge to Tikz ################

data(num_data)
kn <- knowledge(
  num_data,
  X1 \%-->\% X2,
  X2 \%!-->\% c(X3, Y),
  Y \%!-->\% Z
)

# Full standalone document
tikz_kn <- make_tikz(kn, scale = 10, full_doc = TRUE)
cat(tikz_kn)

# Only the tikzpicture environment
tikz_kn_snippet <- make_tikz(kn, full_doc = FALSE)
cat(tikz_kn_snippet)

# With bent edges
tikz_bent <- make_tikz(
  kn,
  full_doc = FALSE,
  bend_edges = TRUE
)
cat(tikz_bent)

# With a color not supported by default TikZ colors; will fall back to RGB
tikz_darkblue <- make_tikz(
  kn,
  node_style = list(fill = "darkblue"),
  full_doc = FALSE
)
cat(tikz_darkblue)

# With tiered knowledge
data(tpc_example)
kn_tiered <- knowledge(
  tpc_example,
  tier(
    child ~ starts_with("child"),
    youth ~ starts_with("youth"),
    old ~ starts_with("old")
  )
)
tikz_tiered_kn <- make_tikz(
  kn_tiered,
  full_doc = FALSE
)
cat(tikz_tiered_kn)


################# Convert disco to Tikz ################

data(num_data)
kn <- knowledge(
  num_data,
  X1 \%-->\% X2,
  X2 \%!-->\% c(X3, Y),
  Y \%!-->\% Z
)

pc_bnlearn <- pc(engine = "bnlearn", test = "fisher_z", alpha = 0.05)
disco_kn <- disco(data = num_data, method = pc_bnlearn, knowledge = kn)

tikz_snippet <- make_tikz(disco_kn, scale = 10, full_doc = FALSE)
cat(tikz_snippet)

################# Convert caugi objects to Tikz ################

cg <- caugi::caugi(A \%-->\% B + C)

tikz_snippet <- make_tikz(
  cg,
  node_style = list(fill = "red"),
  scale = 10,
  full_doc = FALSE
)
cat(tikz_snippet)
}
