
#' @title Plots fitted values of dependent variables
#'
#' @description Plots of fitted values of dependent variables including their 
#' median and percentiles.
#' 
#' @param x an object of class \code{ForecastsPANEL} obtained using the
#' \code{forecast()} function containing posterior draws of 
#' fitted values of dependent variables.
#' @param which_c a positive integer or a character string specifying the country 
#' for which the forecast should be plotted.
#' @param probability a parameter determining the interval to be plotted. The 
#' interval stretches from the \code{0.5 * (1 - probability)} to 
#' \code{1 - 0.5 * (1 - probability)} percentile of the posterior distribution.
#' @param data_in_plot a fraction value in the range (0, 1) determining how many
#' of the last observations in the data should be plotted with the forecasts.
#' @param col a colour of the plot line and the ribbon
#' @param main an alternative main title for the plot
#' @param xlab an alternative x-axis label for the plot
#' @param mar.multi the default \code{mar} argument setting in \code{graphics::par}. Modify with care!
#' @param oma.multi the default \code{oma} argument setting in \code{graphics::par}. Modify with care!
#' @param ... additional arguments affecting the summary produced.
#' 
#' @method plot ForecastsPANEL
#' 
#' @seealso \code{\link{forecast.PosteriorBVARPANEL}}
#'
#' @author Tomasz Woźniak \email{wozniak.tom@pm.me}
#' 
#' @examples
#' specification = specify_bvarPANEL$new(ilo_dynamic_panel[1:5])  # specify the model
#' burn_in       = estimate(specification, 10)               # run the burn-in
#' posterior     = estimate(burn_in, 10)                     # estimate the model
#'
#' # forecast 5 years ahead
#' predictive    = forecast(posterior, 6)
#' plot(predictive, which_c = "ARG")                                # plot forecasts
#' 
#' @export
plot.ForecastsPANEL = function(
    x,
    which_c,
    probability = 0.9,
    data_in_plot = 1,
    col = "#1614B1",
    main,
    xlab,
    mar.multi = c(1, 4.6, 0, 2.1),
    oma.multi = c(6, 0, 5, 0),
    ...
) {
  
  if (is.numeric(which_c)) {
    stopifnot("Argument which_c must be a positive integer indicating one of the countries."
              = length(which_c) == 1 & which_c %% 1 == 0 & which_c > 0 & which_c <= length(x))
  } else if (is.character(which_c)) {
    stopifnot("Argument which_c must be a character string indicating one of the countries."
              = which_c %in% names(x))
  } else {
    stop("Argument which_c must be either a positive integer or a character string.")
  }
  
  if (is.numeric(which_c)) {
    c_name = names(x)[which_c]
  } else if (is.character(which_c)) {
    c_name = which_c
  }
  
  if ( missing(main) ) main = paste("Forecasts for", c_name)
  
  plot(
    x[[which_c]],
    probability,
    data_in_plot,
    col,
    main,
    xlab,
    mar.multi,
    oma.multi,
    ...
  )
} # END plot.ForecastsPANEL








#' @title Plots forecast error variance decompositions
#'
#' @description Plots of the posterior means of the forecast error variance 
#' decompositions.
#' 
#' @param x an object of class \code{PosteriorFEVDPANEL} obtained using the
#' \code{compute_variance_decompositions()} function containing posterior draws of 
#' forecast error variance decompositions.
#' @param which_c a positive integer or a character string specifying the country 
#' for which the forecast should be plotted.
#' @param cols an \code{N}-vector with colours of the plot
#' @param main an alternative main title for the plot
#' @param xlab an alternative x-axis label for the plot
#' @param mar.multi the default \code{mar} argument setting in \code{graphics::par}. Modify with care!
#' @param oma.multi the default \code{oma} argument setting in \code{graphics::par}. Modify with care!
#' @param ... additional arguments affecting the summary produced.
#' 
#' @method plot PosteriorFEVDPANEL
#' 
#' @seealso \code{\link{compute_variance_decompositions.PosteriorBVARPANEL}}
#'
#' @author Tomasz Woźniak \email{wozniak.tom@pm.me}
#' 
#' set.seed(123)
#' specification  = specify_bvarPANEL$new(ilo_dynamic_panel[1:5])
#' 
#' # run the burn-in
#' burn_in        = estimate(specification, 10)
#' 
#' # estimate the model
#' posterior      = estimate(burn_in, 20)
#' 
#' # compute forecast error variance decomposition 4 years ahead
#' fevd           = compute_variance_decompositions(posterior, horizon = 4)
#' plot(fevd, which_c = "ARG")
#' 
#' @export
plot.PosteriorFEVDPANEL = function(
    x,
    which_c,
    cols,
    main,
    xlab,
    mar.multi = c(1, 4.6, 0, 4.6),
    oma.multi = c(6, 0, 5, 0),
    ...
) {
 
  if (is.numeric(which_c)) {
    stopifnot("Argument which_c must be a positive integer indicating one of the countries."
              = length(which_c) == 1 & which_c %% 1 == 0 & which_c > 0 & which_c <= length(x))
  } else if (is.character(which_c)) {
    stopifnot("Argument which_c must be a character string indicating one of the countries."
              = which_c %in% names(x))
  } else {
    stop("Argument which_c must be either a positive integer or a character string.")
  }
  
  N         = dim(x[[which_c]])[1]
  shock_names = 1:N
  
  
  if ( missing(cols) ) {
    fc          = grDevices::colorRampPalette(c("#1A003F", "#1614B1"))
    cols        = fc(N)
  }
  
  if (is.numeric(which_c)) {
    c_name = names(x)[which_c]
  } else if (is.character(which_c)) {
    c_name = which_c
  }
  
  if ( missing(main) ) main = paste("Forecast Error Variance Decompositions for", c_name)
  if ( missing(xlab) ) xlab = "horizon"
  
  plot(
    x[[which_c]],
    shock_names,
    cols,
    main,
    xlab,
    mar.multi,
    oma.multi,
    ...
  )
} # END plot.PosteriorFEVDPANEL
