###### FUNCTION: bpcp2samp ######
## Thanks to Marcel Wolbers for finding this problem.
## It used to produce a warning, but the answers were correct.
## Now should not produce a warning
  sim.exp <- function(n,prob1,prob2,cens.rate){
    # prob1 and prob2 are the absolute risks for an event at time t=1
    rate1 <- -log(1-prob1); rate2 <- -log(1-prob2)
    n1 <- n%/%2+n%%2; n2 <- n%/%2
    ttev.uncens <- rexp(n,rate=rep(c(rate1,rate2),times=c(n1,n2)))
    ttcens <- rexp(n,rate=cens.rate)  
    data.frame(arm=rep(c(0,1),c(n1,n2)),ttev=pmin(ttev.uncens,ttcens),ev=ifelse(ttev.uncens <=ttcens,1,0))
  }
  
  set.seed(1)
  d <- sim.exp(400,prob1=0.1,prob2=0.05,cens.rate=0.05)
  
  test_that("No warning (Problem found by Marcel Wolbers)", {  
  # no warnings
  expect_warning(bpcp2samp(d$ttev,d$ev,d$arm,testtime=1,parmtype="difference"), regexp = NA)
})
  
  # Possibly skip bpcp2sampControl tests - come back
  
  test_that("Error in bpcp2sampControl if stype is not km or mue", {
    
    expect_error(bpcp2sampControl(stype="test"),regexp="stype should be 'km' or 'mue', see bpcp help")
  })
  
  test_that("Error in bpcp2sampControl if eps not between 0 and 0.1", {
    
    expect_error(bpcp2sampControl(eps=1),regexp="eps not in reasonable range, see help for betaMeldTest")
    
    expect_error(bpcp2sampControl(eps=-1),regexp="eps not in reasonable range, see help for betaMeldTest")
  })

  test_that("Error in bpcp2sampControl if Delta is negative", {
    
    expect_error(bpcp2sampControl(Delta=-1),regexp="Delta is width of grouped confidence intervals, must be nonnegative")
    
    expect_error(bpcp2sampControl(Delta=0),regexp=NA)
  })
  
  test_that("Error in bpcp2sampControl if nmc is a negative or non-integer ", {
    
    expect_error(bpcp2sampControl(nmc=0.1),regexp="nmc should be a non-negative integer (but can be numeric)", fixed=TRUE)
    
    expect_error(bpcp2sampControl(nmc=-1),regexp="nmc should be a non-negative integer (but can be numeric)", fixed=TRUE)
  })
  
  test_that("Error in bpcp2sampControl if method is not mm.mc or mc.mc",{
    
    expect_error(bpcp2sampControl(method="test"),
                 regex="method should be 'mm.mc' (method of moments for 1 sample, Monte Carlo for melding) or 'mc.mc' (MC for both) ", fixed=TRUE)
  })
  

  test_that("Error in bpcp2samp if group does not have at least 2 levels",{
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status,group=rep(1,nrow(leuk2)), testtime=9),
                 regexp = "group does not have 2 levels")
  })
  
  
  test_that("Error in bpcp2samp if length of time, status and group are not the same",{
    
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status,group=leuk2$treatment[-1], testtime=9),
                 regexp="length of time, status and group should be the same")
    
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status[-1],group=leuk2$treatment, testtime=9),
                 regexp="length of time, status and group should be the same")
    
    expect_error(bpcp2samp(time=leuk2$time[-1],status=leuk2$status,group=leuk2$treatment, testtime=9),
                 regexp="length of time, status and group should be the same")
    
  })
  
  test_that("Error in bpcp2samp if testime is not positive",{
    
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status,group=leuk2$treatment, testtime=-1),
                 regexp="testtime must be positive and a vector of length 1")
  })
  
  test_that("Error in bpcp2samp if testime is not a vector of length 1",{
    
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status,group=leuk2$treatment, testtime=c(9,13)),
                 regexp="testtime must be positive and a vector of length 1")
  })
  
  test_that("Error in bpcp2samp if alternative is not correctly specified", {
    
    expect_error(bpcp2samp(time=leuk2$time,status=leuk2$status,group=leuk2$treatment, testtime=9,alternative = "test"),
                 regexp = '\'arg\' should be one of "two.sided", "less", "greater"')
  })
  
  test_that("Error in bpcp2samp if add.eps is >= minimum time difference",{
    
    time <- c(1,2,3,5,5)
    status <- c(1,1,1,1,1)
    group <- c("T","C","C","C","T")
    
    expect_error(bpcp2samp(time=time,status=status,group=group, testtime=5, add.eps=1),
                 regexp="Minimum difference between consecutive timepoints is 1. add.eps is greater than or equal to this and needs to be reduced.")
  })
  
  test_that("Warning in bpcp2samp if add.eps is >= 0.5 minimum time difference",{
    
    time <- c(1,2,3,5,5)
    status <- c(1,1,1,1,1)
    group <- c("T","C","C","C","T")
    
    expect_warning(bpcp2samp(time=time,status=status,group=group, testtime=5, add.eps=0.5),
                 regexp="Minimum difference between consecutive timepoints is 1. add.eps is on the same order as this and should be reduced.")
  })


test_that("Two sample test works on differences with no censoring and matches Fisher's exact - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "difference")$conf.int)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "difference")$p.value)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p"), 
               exact2x2::exact2x2(matrix(c(2,1,0,3), nrow=2, ncol=2), tsmethod = "central")$p.value, tolerance = 1e-6)
  
})


test_that("Two sample test works on differences with no censoring and matches Fisher's exact - time 2", {

   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci"),
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "difference")$conf.int)
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p"),
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "difference")$p.value)
  
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p"),
               exact2x2::exact2x2(matrix(c(3,0,0,3), nrow=2, ncol=2), tsmethod = "central")$p.value, tolerance = 1e-6)
  
})


test_that("Two sample test works on differences with no censoring and matches Fisher's exact - time 3", {

   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci"), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "difference")$conf.int)
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p"), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "difference")$p.value)
  
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p"),
               exact2x2::exact2x2(matrix(c(3,0,1,2), nrow=2, ncol=2), tsmethod = "central")$p.value, tolerance = 1e-6)
  

})


test_that("Two sample test works on differences with no censoring and matches Fisher's exact - time 4", {

   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci"), 
               exact2x2::binomMeld.test(0,3,1,3, parmtype = "difference")$conf.int)
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p"), 
               exact2x2::binomMeld.test(0,3,1,3, parmtype = "difference")$p.value)
  
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p"), 
               exact2x2::exact2x2(matrix(c(3,0,2,1), nrow=2, ncol=2), tsmethod = "central")$p.value, tolerance = 1e-6)
  

})


test_that("Two sample test works on differences with no censoring and matches Fisher's exact - time 5", {

   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "difference")$conf.int)
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "difference")$p.value)
  
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p"),
               exact2x2::exact2x2(matrix(c(3,0,3,0), nrow=2, ncol=2), tsmethod = "central")$p.value, tolerance = 1e-6)
  
  
})


# Odds Ratio
test_that("Two sample test works on odds ratios with no censoring - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio")$conf.int)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio")$p.value)
})

test_that("Two sample test works on odds ratios with no censoring - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio")$conf.int)
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio")$p.value)
})

test_that("Two sample test works on odds ratios with no censoring - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio"),
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio")$conf.int)
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio"),
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio")$p.value)
  
})

test_that("Two sample test works on odds ratios with no censoring - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio"),
              exact2x2::binomMeld.test(0,3,1,3, parmtype = "oddsratio")$conf.int)
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio"),
               exact2x2::binomMeld.test(0,3,1,3, parmtype = "oddsratio")$p.value)
})

test_that("Two sample test works on odds ratios with no censoring - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio")$conf.int)
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "oddsratio"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio")$p.value)
  
})

# Ratio
test_that("Two sample test works on ratios with no censoring - time 1", {
   
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "ratio"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "ratio")$conf.int)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "ratio"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "ratio")$p.value)
})

test_that("Two sample test works on ratios with no censoring - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "ratio"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "ratio")$conf.int)
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "ratio"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "ratio")$p.value)
})

test_that("Two sample test works on ratios with no censoring - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "ratio")$conf.int)
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "ratio")$p.value)
  
})

test_that("Two sample test works on ratios with no censoring - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,1,3, parmtype = "ratio")$conf.int)
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,1,3, parmtype = "ratio")$p.value)
})

test_that("Two sample test works on ratios with no censoring - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "ratio")$conf.int)
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "ratio"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "ratio")$p.value)
  
})

test_that("Mid-p for odds ratio with no censoring matches exact2x2 mid-p (Melded Binomial) - time 1", {
   skip_on_cran()
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",midp=T)[1:2], 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",midp=T)$conf.int[1:2], tolerance=0.01)
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",midp=T), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",midp=T)$p.value,tolerance=0.01)
})

test_that("Mid-p for odds ratio with no censoring matches exact2x2 mid-p (Melded Binomial) - time 2", {
  skip_on_cran()
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",midp=T)[1:2], 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",midp=T)$conf.int[1:2], tolerance=0.01)
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",midp=T), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",midp=T)$p.value,tolerance=0.01)
})

test_that("Mid-p for odds ratio with no censoring matches exact2x2 mid-p (Melded Binomial) - time 3", {
  skip_on_cran()
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",midp=T)[1:2], 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",midp=T)$conf.int[1:2], tolerance=0.01)
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",midp=T), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",midp=T)$p.value,tolerance=0.01)
})

test_that("Mid-p for odds ratio with no censoring matches exact2x2 mid-p (Melded Binomial) - time 4", {
  skip_on_cran()
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",midp=T)[1:2], 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",midp=T)$conf.int[1:2], tolerance=0.01)
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",midp=T), 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",midp=T)$p.value,tolerance=0.01)
})

test_that("Mid-p for odds ratio with no censoring matches exact2x2 mid-p (Melded Binomial) - time 5", {
  skip_on_cran()
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",midp=T)[1:2], 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",midp=T)$conf.int[1:2], tolerance=0.01)
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "oddsratio",midp=T), 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",midp=T)$p.value,tolerance=0.01)
})

# conf.level
test_that("90% for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 1", {
   
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",conf.level = 0.9)[1:2], 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",conf.level = 0.9)$conf.int[1:2], tolerance=0.01)
  })

test_that("90% for odds ratio with no censoring matches exact2x2  (Melded Binomial) - time 2", {
  
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",conf.level = 0.9)[1:2], 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",conf.level = 0.9)$conf.int[1:2], tolerance=0.01)
  })

test_that("90% for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",conf.level = 0.9)[1:2], 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",conf.level = 0.9)$conf.int[1:2], tolerance=0.01)
  })

test_that("90% for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",conf.level = 0.9)[1:2], 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",conf.level = 0.9)$conf.int[1:2], tolerance=0.01)
 })

test_that("90% for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",conf.level = 0.9)[1:2], 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",conf.level = 0.9)$conf.int[1:2], tolerance=0.01)
  })


# nullparm
test_that("Nullparm = 0.5 p-value for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",nullparm=0.5)$p.value, tolerance=0.01)
  
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5),
                         no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
})

test_that("Nullparm = 0.5 p-value for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",nullparm=0.5)$p.value, tolerance=0.01)
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5),
                         no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  
  })

test_that("Nullparm = 0.5 p-value for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",nullparm=0.5)$p.value, tolerance=0.01)
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5),
                         no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
})

test_that("Nullparm = 0.5 p-value for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5), 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",nullparm=0.5)$p.value, tolerance=0.01)
  
  expect_false(identical(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5),
                         no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
})

test_that("Nullparm = 0.5 p-value for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "oddsratio",nullparm=0.5), 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",nullparm=0.5)$p.value, tolerance=0.01)
  
})


# alternative

test_that("Alternative = less for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",alternative="less")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2], 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",alternative="less")$conf.int[1:2], tolerance=0.01)
  
  
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"),
                         no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))

  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2],
                         no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
  })

test_that("Alternative = greater for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater"), 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",alternative="greater")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2], 
               exact2x2::binomMeld.test(0,3,2,3, parmtype = "oddsratio",alternative="greater")$conf.int[1:2], tolerance=0.01)
  
  
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater"),
                         no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2],
                         no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
})


test_that("Alternative = greater does not give same results as alternative  = less (time 1, no censoring)", {
   
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"),
                         no_censoring(time=1,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater")))
  
  expect_false(identical(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2],
                         no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2]))
  
})



test_that("Alternative = less for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",alternative = "less")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2], 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",alternative = "less")$conf.int[1:2], tolerance=0.01)
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"),
                         no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2],
                         no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
  
})

test_that("Alternative = greater for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"), 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",alternative = "greater")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2], 
               exact2x2::binomMeld.test(0,3,3,3, parmtype = "oddsratio",alternative = "greater")$conf.int[1:2], tolerance=0.01)
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"),
                         no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2],
                         no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
  
})

test_that("Alternative = greater does not give same results as alternative  = less (time 2, no censoring)", {
   
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"),
                         no_censoring(time=2,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater")))
  
  expect_false(identical(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2],
                         no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2]))
  
})


test_that("Alternative = less for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",alternative = "less")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2], 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",alternative = "less")$conf.int[1:2], tolerance=0.01)
  
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"),
                         no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2],
                         no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
  
})

test_that("Alternative = greater does not give same results as alternative  = less (time 3, no censoring)", {
   
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"),
                         no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater")))
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2],
                         no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2]))
  
})



test_that("Alternative = greater for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"), 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",alternative = "greater")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2], 
               exact2x2::binomMeld.test(1,3,3,3, parmtype = "oddsratio",alternative = "greater")$conf.int[1:2], tolerance=0.01)
  
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"),
                         no_censoring(time=3,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2],
                         no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
  
})

test_that("Alternative = less for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"), 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",alternative = "less")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2], 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",alternative = "less")$conf.int[1:2], tolerance=0.01)
  
})

test_that("Alternative = greater for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"), 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",alternative = "greater")$p.value, tolerance=0.01)
  
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2], 
               exact2x2::binomMeld.test(2,3,3,3, parmtype = "oddsratio",alternative = "greater")$conf.int[1:2], tolerance=0.01)
  
  
  expect_false(identical(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"),
                         no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio")))
  
  expect_false(identical(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2],
                         no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio")[1:2]))
  
})

test_that("Alternative = greater does not give same results as alternative  = less (time 4, no censoring)", {
   
  expect_false(identical(no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="less"),
                         no_censoring(time=4,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative="greater")))
  
  expect_false(identical(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="less")[1:2],
                         no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative="greater")[1:2]))
  
})


test_that("Alternative = less for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "less"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",alternative = "less")$p.value, tolerance=0.01)
  

  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "less")[1:2], 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",alternative = "less")$conf.int[1:2], tolerance=0.01)
  
  
  })

test_that("Alternative = greater for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="p", parmtype = "oddsratio",alternative = "greater"),
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",alternative = "greater")$p.value, tolerance=0.01)
  
  
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",alternative = "greater")[1:2], 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio",alternative = "greater")$conf.int[1:2], tolerance=0.01)
  
  
})

# changeGroupOrder

test_that("changeGroupOrder for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 1", {
  
  expect_equal(no_censoring(time=1,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",changeGroupOrder = T)[1:2], 
               exact2x2::binomMeld.test(2,3,0,3, parmtype = "oddsratio")$conf.int[1:2], tolerance=0.01)
})

test_that("changeGroupOrder for odds ratio with no censoring matches exact2x2  (Melded Binomial) - time 2", {
   
  expect_equal(no_censoring(time=2,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",changeGroupOrder = T)[1:2], 
               exact2x2::binomMeld.test(3,3,0,3, parmtype = "oddsratio")$conf.int[1:2], tolerance=0.01)
})

test_that("changeGroupOrder for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 3", {
   
  expect_equal(no_censoring(time=3,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",changeGroupOrder = T)[1:2], 
               exact2x2::binomMeld.test(3,3,1,3, parmtype = "oddsratio")$conf.int[1:2], tolerance=0.01)
})

test_that("changeGroupOrder for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 4", {
   
  expect_equal(no_censoring(time=4,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",changeGroupOrder = T)[1:2], 
               exact2x2::binomMeld.test(3,3,2,3, parmtype = "oddsratio")$conf.int[1:2], tolerance=0.01)
})

test_that("changeGroupOrder for odds ratio with no censoring matches exact2x2 (Melded Binomial) - time 5", {
   
  expect_equal(no_censoring(time=5,type="bpcp",arms=2,param="ci", parmtype = "oddsratio",changeGroupOrder = T)[1:2], 
               exact2x2::binomMeld.test(0,3,0,3, parmtype = "oddsratio")$conf.int[1:2], tolerance=0.01)
})

test_that("add.eps= 0 gives same results with no censoring", {
  time <- c(1,2,3,5,5)
  status <- c(1,1,1,1,1)
  group <- c("T","C","C","C","T")
  
  expect_equal(bpcp2samp(time,status,group,5,add.eps=0),
              bpcp2samp(time,status,group,5))
})


test_that("add.eps= 0 gives different results than 1e-10 when one group is censored at testtime", {
  # example dataset where all are censored in one group
  set.seed(1)
  n<-10
  y0<- rep(28,n)
  #y0<-c(48*runif(n/2),rep(28+1e-10,n/2))
  y1<- 28*runif(n)
  y<-c(y0,y1)
  #cens<-rep(28,2*n)
  status<- c(rep(0,n),rep(1,n))
  #status<- ifelse(y<=cens,1,0)
  group<- c(rep(0,n),rep(1,n))
  
  # when add.eps = 0, expect p-value = 1
  expect_equal(unname(bpcp2samp(y,status,group,testtime=28,add.eps=0)$p.value), 1)
  
  # when add.eps = 1e-10, expect p-value to be < .001
  expect_lt(unname(bpcp2samp(y,status,group,testtime=28)$p.value), 0.001)
  
  # Also expect upper CI of add.eps = 0 > add.eps = 1e-10
  expect_lt(bpcp2samp(y,status,group,testtime=28)$conf.int[2], bpcp2samp(y,status,group,testtime=28, add.eps=0)$conf.int[2])
  
})

# We expect that if we test a null hypothesis and get a p-value of p
# with a null parameter value of  beta_0
# then the 100(1-p)% confidence interval for beta
# should have one of its limits equal to (or at least very close to) beta_0

test_that("Compatability of p-value and confidence limit (non-extreme data) - difference, less", {
  
  
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"difference","less",0.2),0.2,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - difference, greater", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"difference","greater",0.2),0.2,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - ratio, less ", {
   
  
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"ratio","less",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - ratio, greater", {
  
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"ratio","greater",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - oddsratio, less", {
   
  
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"oddsratio","less",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - oddsratio, greater", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"oddsratio","greater",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - cdfratio, less", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"cdfratio","less",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - cdfratio, greater", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"cdfratio","greater",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - logsratio, less", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"logsratio","less",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - logsratio, greater", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"logsratio","greater",0.3),0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - effcdf, less", {
   
  
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"effcdf","less",1-1/0.3),1-1/0.3,
    tolerance=0.001)
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - effcdf, greater", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"effcdf","greater",1-1/0.3),1-1/0.3,
    tolerance=0.001) 
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data) - efflogs, less", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"efflogs","less",1-1/0.3),1-1/0.3,
    tolerance=0.001)
  
})
test_that("Compatability of p-value and confidence limit (non-extreme data - efflogs, greater)", {
   
  expect_equal(
    pvalue.CI.compatability(3,12,9,24,"efflogs","greater",1-1/0.3),1-1/0.3,
    tolerance=0.001)
  
})

test_that("Compatability of p-value and confidence limit 
               (extreme data) - difference, less)", {
                            
                 # x1=0,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(0,12,9,24,"difference","less",0.2),0.2,
                   tolerance=0.001)
                 
                 # x2=0,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(3,12,0,24,"difference","less",-0.2),-0.2,
                   tolerance=0.001)
                 
                 # x1=n1,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(12,12,9,24,"difference","less",-0.2),-0.2,
                   tolerance=0.001)
                 
                 # x2=n2,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(3,12,24,24,"difference","less",0.9),0.9,
                   tolerance=0.001)
                 
                 # x1=n1, x2=n2, pick nullparm carefully 
                 expect_equal(
                   pvalue.CI.compatability(12,12,24,24,"difference","less",0.1),0.1,
                   tolerance=0.001)
                 
                 # x1=0, x2=0, pick nullparm carefully 
                 expect_equal(
                   pvalue.CI.compatability(0,12,0,24,"difference","less",0.1),0.1,
                   tolerance=0.001)
                 
               })

test_that("Compatability of p-value and confidence limit 
               (extreme data) - difference, greater)", {
                  
                 # x1=0,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(0,12,9,24,"difference","greater",0.2),0.2,
                   tolerance=0.001)
                 
                 # x2=0,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                expect_equal(
                   pvalue.CI.compatability(3,12,0,24,"difference","greater",-0.2),-0.2,
                   tolerance=0.001)
                 
                 # x1=n1,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(12,12,9,24,"difference","greater",-0.9),-0.9,
                   tolerance=0.001)
                 
                 # x2=n2,  cannot expect good tolerance if 
                 # p-value is too small (very close to 0) or too big (very close to 1)
                 # so pick nullparm to avoid those situations
                 expect_equal(
                   pvalue.CI.compatability(3,12,24,24,"difference","greater",0.5),0.5,
                   tolerance=0.001)
                 
                 # x1=n1, x2=n2, pick nullparm carefully 
                 expect_equal(
                   pvalue.CI.compatability(12,12,9,24,"difference","greater",-0.1),-0.1,
                   tolerance=0.001)
                 
                 # x1=0, x2=0, pick nullparm carefully 
                 expect_equal(
                   pvalue.CI.compatability(0,12,0,24,"difference","greater",-0.1),-0.1,
                   tolerance=0.001) 
               })

test_that("Compatability of p-value and confidence limit 
               (extreme data) - ratio, less", {
  
                  
  # ratio
  # x1=0 and x2=0 gives no information about ratio parameters
  # so we expect p.value=1, conf.int=(0,Inf) [for ratio]
  # Estimates= NaN
  x<- bpcp2samp.nocensoring(0,12,0,24,"ratio","less",1)
  expect_equal(unname(c(x$p.value,x$conf.int,x$estimate)),c(1,0,Inf,NA))
  
               })

test_that("Compatability of p-value and confidence limit 
               (extreme data) - cdfratio, less", {
  # cdfratio
  # x1=n1 and x2=n2 gives no information about cdfratio parameters
  # so we expect p.value=1, conf.int=(0,Inf) [for cdfratio]
  # Estimates= NaN
  x<- bpcp2samp.nocensoring(12,12,24,24,"cdfratio","less",1)
  expect_equal(unname(c(x$p.value,x$conf.int,x$estimate)),c(1,0,Inf,NA))
  
               })

test_that("Compatability of p-value and confidence limit 
               (extreme data) - logsratio, less", {
                  
  # logsratio
  # EITHER x1=n1 and x2=n2 OR  x1=0 and x2=0  gives no information 
  # about logsratio parameters
  # so we expect p.value=1, conf.int=(0,Inf) [for cdfratio]
  # Estimates= NaN
  x<- bpcp2samp.nocensoring(12,12,24,24,"logsratio","less",1)
  expect_equal(unname(c(x$p.value,x$conf.int,x$estimate)),c(1,0,Inf,NA))
  
  x<- bpcp2samp.nocensoring(0,12,0,24,"logsratio","less",1)
  expect_equal(unname(c(x$p.value,x$conf.int,x$estimate)),c(1,0,Inf,NA))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = difference, melded", {
  
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                   parmtype = "difference")
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = difference, melded with midp", {
  skip_on_cran()
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "difference", midp = T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = oddsratio, melded", {
   
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                     parmtype = "oddsratio", midp = F)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = oddsratio, melded with midp", {
  
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                   parmtype = "oddsratio",midp = T, testtime=20)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = ratio, melded", {
   
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "ratio", midp = F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = ratio, melded with midp", {
  skip_on_cran()
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "ratio", midp = T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})


test_that("All CI for twosamp output contain beta estimate, parmtype = efflogs, melded", {
   
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "efflogs", midp = F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = efflogs, melded with midp", {
  skip_on_cran()
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "efflogs", midp = T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})


test_that("All CI for twosamp output contain beta estimate, parmtype = effcdf, melded", {
   
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "effcdf", midp = F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("All CI for twosamp output contain beta estimate, parmtype = effcdf, melded with midp", {
  skip_on_cran()
  res <- bpcp2samp(time=leuk2$time, status=leuk2$status, group=leuk2$treatment,
                    parmtype = "effcdf", midp = T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

# S1 = s, S2 = 1
test_that("CI contains beta estimate, parmtype = difference, melded, S1= 1, S2=s", {
  
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "difference", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= 1, S2=s", {
  
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "oddsratio", midp=F)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "oddsratio", midp=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

# S1 = 0, S2 = 1 

test_that("CI contains beta estimate, parmtype = difference, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                  parmtype = "difference", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                    parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "difference", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                  parmtype = "ratio", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= 1, S2=0", {
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= 0, S2=1", {
  
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "oddsratio", midp=F)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                  parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "oddsratio", midp=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                    parmtype = "effcdf", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                    parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "effcdf", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "effcdf", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= 0, S2=1", {
  
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= 0, S2=1", {
  skip_on_cran()
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})



# S1 = s, S2 = 0

test_that("CI contains beta estimate, parmtype = difference, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = difference, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                    parmtype = "ratio", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "ratio", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = ratio, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= 0, S2=s", {
  
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                    parmtype = "oddsratio", midp=F)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "oddsratio", midp=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = oddsratio, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower[!is.na(res$gbeta)] <= res$gbeta[!is.na(res$gbeta)]) & 
                all(res$upper[!is.na(res$gbeta)] >= res$gbeta[!is.na(res$gbeta)]))
  
  expect_true(all(res$lower[is.na(res$gbeta)] == 0) & 
                all(res$upper[is.na(res$gbeta)] == Inf))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                  parmtype = "effcdf", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                  parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "effcdf", midp=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = effcdf, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "effcdf", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=F)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=T, testtime=7)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})

test_that("CI contains beta estimate, parmtype = efflogs, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
  expect_true(all(res$lower <= res$gbeta) & all(res$upper >= res$gbeta))
  
})


test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=F)
  
  
  expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
  expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})


test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "oddsratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "oddsratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= s, S2=1", {
  skip_on_cran() 
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= 1, S2=s", {
  
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                  parmtype = "effcdf", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "effcdf", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= 1, S2=s", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                    parmtype = "efflogs", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= s, S2=1", {
   
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                  parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= 1, S2=s", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= s, S2=1", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure$time, status=d_failure$status, group=d_failure$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

# S1 = 0, S2 = 1 

test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                    parmtype = "difference", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "difference", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "ratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "oddsratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "oddsratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= 0, S2=1", {
   
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "effcdf", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= 0, S2=1", {
  
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "effcdf", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "effcdf", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= 0, S2=1", {
  
  
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "efflogs", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= 1, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= 0, S2=1", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= 1, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

# S1 = s, S2 = 0

test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "difference", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "difference", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= 0, S2=s", {
  skip_on_cran()
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = difference, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "difference", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "ratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                     parmtype = "ratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "ratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = ratio, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_2$time, status=d_failure_2$status, group=d_failure_2$rx,
                   parmtype = "ratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= 0, S2=s", {
  
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "oddsratio", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "oddsratio", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "oddsratio", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = oddsratio, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "oddsratio", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "effcdf", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "effcdf", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "effcdf", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = effcdf, melded with midp, S1= s, S2=0", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "effcdf", midp=T,changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})


test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= 0, S2=s", {
   
  
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=F)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded, S1= s, S2=0", {
   
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                     parmtype = "efflogs", midp=F, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})



test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= 0, S2=s", {
   
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})

test_that("p-value is between 0 and 1, parmtype = efflogs, melded with midp, S1= s, S2=0", {
  skip_on_cran()
  res <- bpcp2samp(time=d_failure_3$time, status=d_failure_3$status, group=d_failure_3$rx,
                   parmtype = "efflogs", midp=T, changeGroupOrder=T)
  
  
   expect_true(all(res$p.value <= 1) & all(res$p.value >= 0))
  
})