% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blatentSimulate.R
\name{blatentSimulate}
\alias{blatentSimulate}
\title{Simulates data using blatent syntax and simulated parameters input}
\usage{
blatentSimulate(
  modelText,
  nObs,
  defaultSimulatedParameters = setDefaultSimulatedParameters(),
  paramVals = NULL,
  seed = NULL,
  calculateInfo = FALSE
)
}
\arguments{
\item{modelText}{A character string that contains the specifications for the model to be run. See \code{\link{blatentSyntax}}
or more information about syntax formatting.}

\item{nObs}{The number of observations to be simulated.}

\item{defaultSimulatedParameters}{The specifications for the generation of the types of parameters in the simulation. Currently comprised
of a list of unevaluated expressions (encapsulated in quotation marks; not calls for ease of user input) that will be evaluated by
simulation function to generate parameters. Defaults to values generated by \code{\link{setDefaultSimulatedParameters}}.
The list of unevaluated expressions must include:
\itemize{
  \item \code{observedIntercepts} The data generating function for all intercepts for observed variables.
  \item \code{observedMainEffects} The data generating function for the main effects for observed variables.
  \item \code{observedInteractions} The data generating function for all interactions for observed variables.
  \item \code{latentIntercepts} The data generating function for all intercepts for latent variables.
  \item \code{latentMainEffects} The data generating function for the main effects for latent variables.
  \item \code{latentInteractions} The data generating function for all interactions for latent variables.
}}

\item{paramVals}{A named vector of parameter values which will be set rather than generated. A named vector of the length parameters of an analysis
can be obtained by using \code{\link{createParameterVector}}. The NA values of this vector can be overwritten by values to be used in the
simulation.}

\item{seed}{The random number seed value used for setting the data. Defaults to \code{NULL}.}

\item{calculateInfo}{A logical variable where \code{TRUE} indicates information statistics will be calculated (only when a single latent
multivariate Bernoulli variable is in the model) and \code{FALSE} disables calculation.}
}
\description{
Simulates data from a model specified by blatent syntax and using a set of default parameter specifications.
}
\examples{

# Generating data using Q-matrix structure from data example in Chapter 9 of
#  Rupp, Templin, & Henson (2010).

RTHCh9ModelSyntax = "
   item1 ~ A1
   item2 ~ A2
   item3 ~ A3
   item4 ~ A1 + A2 + A1:A2
   item5 ~ A1 + A3 + A1:A3
   item6 ~ A2 + A3 + A2:A3
   item7 ~ A1 + A2 + A3 + A1:A2 + A1:A3 + A2:A3 + A1:A2:A3

   # Latent Variable Specifications:
 A1 A2 A3 <- latent(unit='rows',distribution='bernoulli',structure='univariate',type='ordinal')

   # Observed Variable Specifications:
   item1-item7 <- observed(distribution = 'bernoulli', link = 'probit')
"

simSpecs = setDefaultSimulatedParameters(
   observedIntercepts = "runif(n = 1, min = -1, max = -1)",
   observedMainEffects = "runif(n = 1, min = 2, max = 2)",
   observedInteractions = "runif(n = 1, min = 0, max = 0)",
   latentIntercepts = "runif(n = 1, min = 0, max = 0)",
   latentMainEffects  = "runif(n = 1, min = 0, max = 0)",
   latentInteractions = "runif(n = 1, min = 0, max = 0)"
 )

simulatedData = blatentSimulate(modelText = RTHCh9ModelSyntax, nObs = 1000,
                                defaultSimulatedParameters = simSpecs)

# setting values for specific parameters:
paramVals = createParameterVector(modelText = RTHCh9ModelSyntax)
paramVals["item1.(Intercept)"] = -2

# creating data
simulatedData2 = blatentSimulate(modelText = RTHCh9ModelSyntax, nObs = 1000,
                                 defaultSimulatedParameters = simSpecs, paramVals = paramVals)

}
\references{
Rupp, A. A., Templin, J., & Henson, R. A. (2010). Diagnostic Measurement: Theory, Methods, and Applications. New York: Guilford.
}
