% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trajectories_core.R
\name{filter_trajectories}
\alias{filter_trajectories}
\title{Filter and rank detected trajectories}
\usage{
filter_trajectories(tr_tbl, top_n = 3, min_score = NULL, min_length = NULL)
}
\arguments{
\item{tr_tbl}{A tibble of trajectories from \code{detect_main_trajectories()$trajectories}.
Must contain at least \code{traj_id}, \code{score}, and \code{length} columns.}

\item{top_n}{Maximum number of trajectories to keep after filtering and sorting
(default: 3). If \code{NULL}, keeps all trajectories that meet the filter criteria.}

\item{min_score}{Minimum score threshold for trajectories (default: \code{NULL}).
Trajectories with score less than \code{min_score} are discarded. Useful for
removing weak or noisy paths.}

\item{min_length}{Minimum trajectory length in distinct years (default: \code{NULL}).
Trajectories shorter than \code{min_length} are discarded. Ensures only trajectories
spanning a meaningful temporal horizon are kept.}
}
\value{
A filtered and sorted trajectory tibble with the same structure as input,
containing only trajectories that meet all criteria, sorted by descending score.
Returns an empty tibble if no trajectories meet the criteria.
}
\description{
Applies post-processing filters and ranking to trajectory data based on score,
length, and other criteria. This function helps refine the output from
\code{detect_main_trajectories()} by keeping only the most relevant trajectories
according to user-specified constraints.
}
\details{
This function provides a straightforward way to refine trajectory detection results
by applying quality filters and ranking. The filtering process occurs in three steps:
\enumerate{
\item \strong{Quality Filtering}: Remove trajectories that don't meet minimum quality standards
\itemize{
\item \code{min_score}: Filters by the dynamic programming path score (higher = better)
\item \code{min_length}: Filters by temporal span in distinct years
}
\item \strong{Ranking}: Sort remaining trajectories by descending score to prioritize the
most significant paths
\item \strong{Selection}: Keep only the top \code{top_n} trajectories after filtering and sorting
}
\subsection{Typical Use Cases}{
\itemize{
\item \strong{Focus on strongest signals}: Use \code{min_score} to remove low-confidence trajectories
\item \strong{Ensure temporal significance}: Use \code{min_length} to require multi-year evolution
\item \strong{Limit visualization complexity}: Use \code{top_n} to focus on the most important paths
\item \strong{Progressive refinement}: Chain multiple calls with different criteria
}
}
}
\examples{
\dontrun{
# Get trajectories first
traj_data <- detect_main_trajectories(
  groups_cumulative_trajectories = my_data,
  group = "component1_g01"
)

# Basic: Keep top 3 trajectories by score
top_trajectories <- filter_trajectories(traj_data$trajectories)

# Keep top 5 trajectories with minimum quality standards
quality_trajectories <- filter_trajectories(
  tr_tbl = traj_data$trajectories,
  top_n = 5,
  min_score = 10,
  min_length = 4
)

# Keep all trajectories meeting minimum length (no top_n limit)
long_trajectories <- filter_trajectories(
  tr_tbl = traj_data$trajectories,
  top_n = NULL,
  min_length = 5
)

# Very strict filtering for high-quality, long trajectories
strict_trajectories <- filter_trajectories(
  tr_tbl = traj_data$trajectories,
  top_n = 3,
  min_score = 15,
  min_length = 6
)

# Use filtered trajectories for visualization
plot_group_trajectories_lines_2d(
  traj_data = traj_data,
  traj_filtered = quality_trajectories
)
}

}
\seealso{
\code{\link[=detect_main_trajectories]{detect_main_trajectories()}} for generating the trajectory data,
\code{\link[=plot_group_trajectories_lines_2d]{plot_group_trajectories_lines_2d()}} and \code{\link[=plot_group_trajectories_lines_3d]{plot_group_trajectories_lines_3d()}}
for visualizing filtered trajectories
}
