% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_Report.R
\name{BIOMOD_Report}
\alias{BIOMOD_Report}
\title{Produce summary outputs from a simulation folder}
\usage{
BIOMOD_Report(
  bm.out,
  strategy = "report",
  params.color = list(color1 = "#eb4034", color2 = "#e0918b", color3 = "#658f70"),
  params.ODMAP = list(O.mod.objective = NULL, O.boundary = NULL, O.obs.type = NULL,
    O.pred.type = NULL, D.eco.level = NULL, D.samp.design = NULL)
)
}
\arguments{
\item{bm.out}{a \code{\link{BIOMOD.formated.data}} or \code{\link{BIOMOD.formated.data.PA}} 
object returned by the \code{\link{BIOMOD_FormatingData}} function ; or a 
\code{\link{BIOMOD.models.out}} or \code{\link{BIOMOD.ensemble.models.out}} 
object that can be obtained with the \code{\link{BIOMOD_Modeling}} or 
\code{\link{BIOMOD_EnsembleModeling}} functions}

\item{strategy}{a \code{character} defining the type of summary file that will be produced, 
must be \code{report}, \code{ODMAP} or \code{code} (see Details)}

\item{params.color}{a \code{list} containing 3 color values to custom the reports}

\item{params.ODMAP}{a \code{list} containing values of some ODMAP fields to be filled in 
from pre-existing choices (see Details)}
}
\value{
A standardized \code{.html} file obtained from an \code{Rmarkdown} template, and a \code{.csv} 
table in the case of ODMAP report.
}
\description{
This function allows to produce summary report or ODMAP table from a 
\pkg{biomod2} simulation folder.
}
\details{
This function gathers and formats all objects contained in one \pkg{biomod2} modeling folder 
to produce, based on \code{Rmarkdown} templates, standardized reports to help the user :
\itemize{
  \item summarize its modeling results
  \item share them through standardized informations through ODMAP protocol
  \item provide reproducible code \cr \cr
}


\describe{
  \item{Type of report}{
  Different data types are available, and require different values :
  \describe{
    \item{report}{\pkg{biomod2} provides functions to summarize the information, such as 
    \code{print}, \code{plot} or \code{summary} methods adapted to \code{BIOMOD.[...].out} 
    objects, as well as 
    \href{https://biomodhub.github.io/biomod2/reference/getters.out.html}{\code{get_[...]}} 
    and \code{bm_Plot[...]} functions. All these are called here and applied to objects 
    contained in the provided modeling folder.
    }
    \item{ODMAP}{following Zurell et al. 2020, ODMAP (Overview, Data, Model, Assessment and 
    Prediction) protocol aims to standardize documentation of modeling to help improve both 
    transparency and reproducibility of results. 
    \href{https://odmap.wsl.ch/}{ODMAP v1.0 website} provides an application to fill this type 
    of report. \pkg{biomod2} tries here to help one user to pre-fill the fields of this 
    protocol.
    }
    \item{code}{\code{call} slot contained within \code{\link{BIOMOD.formated.data}}, 
    \code{BIOMOD.models.out}, \code{BIOMOD.projection.out} and 
    \code{BIOMOD.ensemble.models.out} objects keep in memory the R command used to obtain 
    them. All these calls are gathered here in one summary file.}
  }
  }
}
}
\examples{
library(terra)

# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

 
# --------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.name = myRespName,
                                       resp.var = myResp,
                                       resp.xy = myRespXY,
                                       expl.var = myExpl)

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      CV.strategy = 'random',
                                      CV.nb.rep = 2,
                                      CV.perc = 0.8,
                                      OPT.strategy = 'bigboss',
                                      metric.eval = c('TSS', 'AUCroc'),
                                      var.import = 3,
                                      seed.val = 42)
}


file.proj <- paste0(myRespName, "/proj_Current/", myRespName, ".Current.projection.out")
if (file.exists(file.proj)) {
  myBiomodProj <- get(load(file.proj))
} else {

  # Project single models
  myBiomodProj <- BIOMOD_Projection(bm.mod = myBiomodModelOut,
                                    proj.name = 'Current',
                                    new.env = myExpl,
                                    models.chosen = 'all',
                                    build.clamping.mask = TRUE)
}


file.EM <- paste0(myRespName, "/", myRespName, ".AllModels.ensemble.models.out")
if (file.exists(file.EM)) {
  myBiomodEM <- get(load(file.EM))
} else {

  # Model ensemble models
  myBiomodEM <- BIOMOD_EnsembleModeling(bm.mod = myBiomodModelOut,
                                        models.chosen = 'all',
                                        em.by = 'all',
                                        em.algo = c('EMmean', 'EMca'),
                                        metric.select = c('TSS'),
                                        metric.select.thresh = c(0.7),
                                        metric.eval = c('TSS', 'AUCroc'),
                                        var.import = 3,
                                        seed.val = 42)
}


# --------------------------------------------------------------- #
# Compile summary reports
# BIOMOD_Report(bm.out = myBiomodModelOut, strategy = 'report')
# BIOMOD_Report(bm.out = myBiomodProj, strategy = 'report')
# BIOMOD_Report(bm.out = myBiomodEM, strategy = 'report')
 
# BIOMOD_Report(bm.out = myBiomodModelOut, strategy = 'ODMAP')
# BIOMOD_Report(bm.out = myBiomodModelOut, strategy = 'code')



}
\references{
\itemize{
  \item Zurell D, Franklin J, König C, Bouchet PJ, Serra-Diaz JM, Dormann CF, Elith J, 
  Fandos Guzman G, Feng X, Guillera-Arroita G, Guisan A, Leitão PJ, Lahoz-Monfort JJ, 
  Park DS, Peterson AT, Rapacciuolo G, Schmatz DR, Schröder B, Thuiller W, Yates KL, 
  Zimmermann NE, Merow C (\bold{2020}). \emph{A standard protocol for describing species 
  distribution models.} Ecography 43: 1261-1277. \doi{10.1111/ecog.04960}
}
}
\seealso{
\code{\link{ODMAP}}, \code{\link[base]{match.call}}
}
\author{
Maya Guéguen
}
\concept{Primary functions}
\keyword{ODMAP}
\keyword{html}
\keyword{markdown}
\keyword{report}
