% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_varcov.R
\name{estimate_varcov}
\alias{estimate_varcov}
\title{Estimate variance-covariance matrix for marginal estimand based on GLM model}
\usage{
estimate_varcov(
  object,
  strata = NULL,
  method = c("Ge", "Ye"),
  type = c("HC0", "model-based", "HC3", "HC", "HC1", "HC2", "HC4", "HC4m", "HC5"),
  mod = FALSE
)
}
\arguments{
\item{object}{a fitted \code{\link[stats]{glm}} object augmented with
\code{counterfactual.predictions}, \code{counterfactual.predictions} and \code{counterfactual.means}}

\item{strata}{an optional string or vector of strings specifying the names
of stratification variables. Relevant only for Ye's method and used to
adjust the variance-covariance estimation for stratification.
If provided, each specified variable must be present in the model.}

\item{method}{a string indicating the chosen method for variance estimation.
Supported methods are \code{Ge} and \code{Ye}. The default method is \code{Ge} based on Ge
et al (2011) which is suitable for the variance estimation of conditional
average treatment effect. The method \code{Ye} is based on Ye et al (2023) and is
suitable for the variance estimation of population average treatment effect.
For more details, see \href{https://osf.io/9mp58/}{Magirr et al. (2024)}.}

\item{type}{a string indicating the type of
variance estimator to use (only applicable for Ge's method). Supported types include HC0 (default),
model-based, HC3, HC, HC1, HC2, HC4, HC4m, and HC5. See \link[sandwich]{vcovHC} for heteroscedasticity-consistent estimators.
This parameter allows for flexibility in handling heteroscedasticity
and model specification errors.}

\item{mod}{For Ye's method, the implementation of open-source RobinCar package
has an additional variance decomposition step when estimating the robust variance,
which then utilizes different counterfactual outcomes than the original reference.
Set \code{mod = TRUE} to use exactly the implementation method described in
Ye et al (2022), default to \code{FALSE} to use the modified implementation in
RobinCar and Bannick et al (2023) which improves stability.}
}
\value{
an updated \code{glm} object appended with an
additional component \code{robust_varcov}, which is the estimated variance-covariance matrix
of the marginal effect. The matrix format and estimation method are
indicated in the matrix attributes.
}
\description{
Main variance estimation function. Estimates the variance-covariance
matrix of a marginal estimand for a generalized linear model (GLM) object
using specified methods. This function supports both Ge's and Ye's methods
for variance estimation, accommodating different estimand specifications.
}
\details{
The \code{estimate_varcov} function facilitates robust variance estimation
techniques for GLM models, particularly useful in clinical trial analysis
and other fields requiring robust statistical inference. It allows
researchers to account for complex study designs,
including stratification and different treatment contrasts,
by providing a flexible interface for variance-covariance estimation.

Note: Ensure that the \code{glm} object has been adequately prepared with
\code{\link{predict_counterfactuals}} and \code{\link{average_predictions}}
before applying \code{estimate_varcov()}. Failure to do so may result in
errors indicating missing components.
}
\examples{
# Example usage with a binary outcome GLM model
trial01$trtp <- factor(trial01$trtp)
fit1 <- glm(aval ~ trtp + bl_cov, family = "binomial", data = trial01)

#' # Preprocess fit1 as required by estimate_varcov
fit2 <- fit1 |>
  predict_counterfactuals(trt = "trtp") |>
  average_predictions()

# Estimate variance-covariance using Ge's method
fit3_ge <- estimate_varcov(fit2, method = "Ge")
print(fit3_ge$robust_varcov)


# Estimate variance-covariance using Ye's method with stratification
fit4 <- glm(aval ~ trtp + bl_cov_c, family = "binomial", data = trial01) |>
  predict_counterfactuals(trt = "trtp") |>
  average_predictions()
fit4_ye <- estimate_varcov(fit4, method = "Ye", strata = "bl_cov_c")
print(fit4_ye$robust_varcov)

}
\references{
Ye T. et al. (2023) Robust variance
estimation for covariate-adjusted unconditional treatment effect in randomized
clinical trials with binary outcomes. Statistical Theory and Related Fields

Ge M. et al. (2011) Covariate-Adjusted
Difference in Proportions from Clinical Trials Using Logistic Regression
and Weighted Risk Differences. Drug Information Journal.

Bannick, M. S., et al. A General Form of Covariate Adjustment in
Randomized Clinical Trials. arXiv preprint arXiv:2306.10213 (2023).
}
\seealso{
\code{\link[=average_predictions]{average_predictions()}} for averaging counterfactual
predictions.

\code{\link[=apply_contrast]{apply_contrast()}} for computing a summary measure.

\code{\link[=get_marginal_effect]{get_marginal_effect()}} for estimating marginal effects directly
from an original \code{\link[stats]{glm}} object
}
