% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extractors.R
\name{extract_posterior_summary}
\alias{extract_posterior_summary}
\title{Extract posterior summary from fitted model}
\usage{
extract_posterior_summary(fit)
}
\arguments{
\item{fit}{Fitted Stan model object (CmdStanMCMC or stanfit)}
}
\value{
List with components:
  \describe{
    \item{beta1}{Median of global TMG effect}
    \item{beta0_s}{Vector of sector-specific intercepts}
    \item{kappa_s}{Vector of mean reversion speeds}
    \item{a3_s}{Vector of cubic drift coefficients}
    \item{theta_s}{Vector of equilibrium levels}
    \item{rho_s}{Vector of SV persistence parameters}
    \item{alpha_s}{Vector of SV level parameters}
    \item{sigma_eta_s}{Vector of SV volatility parameters}
    \item{nu}{Degrees of freedom for Student-t}
    \item{gamma}{COM effect in mean}
    \item{rhat}{R-hat convergence diagnostics}
    \item{ess}{Effective sample sizes}
  }
}
\description{
Extracts median point estimates and credible intervals for all
model parameters from a fitted Stan model.
}
\examples{
\donttest{
# 1. Create a mock CmdStanMCMC object
# We simulate a posterior distribution for 2 sectors
S <- 2
n_draws <- 100

# Helper to generate random draws
mock_draws <- function(name, n_cols=1) {
  m <- matrix(rnorm(n_draws * n_cols), nrow = n_draws, ncol = n_cols)
  if (n_cols > 1) {
    colnames(m) <- paste0(name, "[", 1:n_cols, "]")
  } else {
    colnames(m) <- name
  }
  as.data.frame(m)
}

# Combine draws into one data frame
df_draws <- cbind(
  mock_draws("beta1", 1),
  mock_draws("beta0_s", S),
  mock_draws("kappa_tilde", S), # Note: function expects log-scale kappa
  mock_draws("a3_tilde", S),    # Note: function expects log-scale a3
  mock_draws("theta_s", S),
  mock_draws("rho_s", S),
  mock_draws("alpha_s", S),
  mock_draws("sigma_eta_s", S),
  mock_draws("nu_tilde", 1),
  mock_draws("gamma", 1)
)

# Mock fit object
mock_fit <- structure(list(
  draws = function(vars, format="df") {
     # Simple regex matching for the mock
     if (length(vars) == 1) {
       # Check if it's a scalar or vector parameter request
       if (vars \%in\% names(df_draws)) return(df_draws[vars])
       # Pattern match for vectors like "beta0_s" -> "beta0_s[1]", "beta0_s[2]"
       cols <- grep(paste0("^", vars, "\\\\["), names(df_draws), value = TRUE)
       if (length(cols) > 0) return(df_draws[cols])
     }
     return(df_draws) 
  },
  summary = function() {
    data.frame(
      variable = names(df_draws),
      rhat = rep(1.0, ncol(df_draws)),
      ess_bulk = rep(400, ncol(df_draws))
    )
  }
), class = "CmdStanMCMC")

# 2. Run extraction
summ <- extract_posterior_summary(mock_fit)
print(summ$kappa_s)
}

}
