% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/acs.R
\name{get_prioritised_covariates}
\alias{get_prioritised_covariates}
\title{Generate the prioritised covariates from the global list of binary recurrence covariates using multiplicative bias ranking}
\usage{
get_prioritised_covariates(
  df,
  patientIdVarname,
  exposureVector,
  outcomeVector,
  patientIdVector,
  k = 500
)
}
\arguments{
\item{df}{The input \code{data.frame}. Ideally this should be the output \code{recurrence_data} from the
\code{\link[autoCovariateSelection]{get_recurrence_covariates}} function}

\item{patientIdVarname}{The variable name which contains the patient identifier in the \code{df}}

\item{exposureVector}{The 1-D exposure (treatment/intervention) vector. The length of this vector should be equal to that of
\code{patientIdVector} and \code{outcomeVector}. Also, this should be a binary vector with value of 1 for patients primary cohort 1 and 0 for
those in comparator cohort. The order of this vector should resonate the order of patients in \code{outcomeVector} and \code{patientIdVector}}

\item{outcomeVector}{The 1-D outcome vector indicating whether or not the patient experienced the outcome of interest (value = 1) or not (value =0).
The length of this vector should be equal to that of \code{patientIdVector} and \code{exposureVector}. The order of elements in this vector should
resonate with the order of patients in \code{exposureVector} and \code{patientIdVector}}

\item{patientIdVector}{The 1-D vector with all the patient identifiers. This should contain all the patient IDs in the original two
cohorts with its length and order equal to and resonating with that of \code{exposureVector} and \code{outcomeVector}}

\item{k}{The maximum number of prioritised covariates that should be returned by the function. By default, this is 500 as described in the original paper}
}
\value{
A named list containing two R objects
\itemize{
\item \code{autoselected_covariate_df} A \code{data.frame} in wide format containing the auto-selected prioritised covariates and their values (1 or 0)
for each patients
\item \code{multiplicative_bias}{The absolute log of the multiplicative bias term for each of the auto-selected prioritised covariates}
}
}
\description{
\code{get_prioritised_covariates} function assesses the recurrence of each of the identified candidate empirical covariates
based on their frequency of occurrence for each patient in the baseline period and generates three binary recurrence covariates
for each of the identified candidate empirical covariates. This is the third and final step in the automated covariate selection process.
The previous step of assessing recurrence and generating the binary recurrence covariates is done
using the \code{\link[autoCovariateSelection]{get_recurrence_covariates}} function.
See 'Automated Covariate Selection'section below for more details regarding the overall process.
}
\details{
To prioritise covariates across data dimensions (domains) should be assessed by their potential for controlling confounding that is not conditional
on exposure and other covariates. This means that the association of the covariates with the outcomes (relative risk) should be taken into
consideration for quantifying the 'potential' for confounding. Relative risk weighted by the ratio of prevalence of the covariates between the
two exposure groups is known as multiplicative bias. The other way to do this would be to use the absolute risk and this would have been the rather
straight-forward procedure to quantify the potential for confounding. However, this method would invariably down-weight the association between the
covariate and the outcome if the outcome prevalence is small and the exposure prevalence is high which is a common phenomenon seen with comparative
effective research using real-world-data by retrospective cohort studies. The multiplicative bias term balances this and generates a quantity for each
covariate that is reflective of its confounding potential. By ranking the multiplicative bias, the objective is to choose the top \code{k} number of
covariates from this procedure. \code{k}, by default, is 500 as described in the original paper. For further theoretical details of the
algorithm please refer to the original article listed below in the \code{References} section. \code{get_recurrence_covariates} is the function
implementing what is described in the 'Prioritise Covariates' section of the article.
}
\section{Automated Covariate Selection}{

\strong{The three steps in automated covariate selection are listed below with the functions implementing the methodology}
\enumerate{
\item Identify candidate empirical covariates: \code{\link[autoCovariateSelection]{get_candidate_covariates}}
\item Assess recurrence: \code{\link[autoCovariateSelection]{get_recurrence_covariates}}
\item Prioritize covariates: \code{\link[autoCovariateSelection]{get_prioritised_covariates}}
}
}

\examples{
library("autoCovariateSelection")
data(rwd)
head(rwd, 3)
basetable <- rwd \%>\% select(person_id, treatment, outcome_date) \%>\% distinct()
head(basetable, 3)
patientIds <- basetable$person_id
step1 <- get_candidate_covariates(df = rwd,  domainVarname = "domain",
eventCodeVarname = "event_code" , patientIdVarname = "person_id",
patientIdVector = patientIds,n = 100, min_num_patients = 10)
out1 <- step1$covars_data
all.equal(patientIds, step1$patientIds) #should be TRUE
step2 <- get_recurrence_covariates(df = out1,
patientIdVarname = "person_id", eventCodeVarname = "event_code",
patientIdVector = patientIds)
out2 <- step2$recurrence_data
out3 <- get_prioritised_covariates(df = out2,
patientIdVarname = "person_id", exposureVector = basetable$treatment,
outcomeVector = ifelse(is.na(basetable$outcome_date), 0,1),
patientIdVector = patientIds, k = 10)

}
\references{
Schneeweiss S, Rassen JA, Glynn RJ, Avorn J, Mogun H, Brookhart MA. High-dimensional propensity score adjustment in studies of treatment effects using health care claims data Epidemiology. 2009;20(4):512-522. doi:10.1097/EDE.0b013e3181a663cc
}
\author{
Dennis Robert \email{dennis.robert.nm@gmail.com}
}
