% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/construct_confidence_set.R
\name{CS.argmin}
\alias{CS.argmin}
\title{Construct a discrete confidence set for argmin.}
\usage{
CS.argmin(data, method = "softmin.LOO", alpha = 0.05, ...)
}
\arguments{
\item{data}{A n by p data matrix; each row is a p-dimensional sample.}

\item{method}{A string indicating the method used to construct the confidence set. Defaults to 'softmin.LOO'.
Can be abbreviated (e.g., 'SML' for 'softmin.LOO'). See **Details** for available methods and abbreviations.}

\item{alpha}{The significance level; defaults to 0.05. The function produces a \eqn{1 - \alpha} confidence set.}

\item{...}{Additional arguments to \link{argmin.HT.LOO}, \link{lambda.adaptive.enlarge}, \link{is.lambda.feasible.LOO}, \link{argmin.HT.MT}, \link{argmin.HT.gupta}.
A correct argument name needs to be specified if it is used.}
}
\value{
A vector of indices (1-based) representing the (1 - alpha) confidence set.
}
\description{
This is a wrapper to construct a discrete confidence set for argmin. Multiple methods are supported.
}
\details{
The supported methods include:\tabular{ll}{
  \code{softmin.LOO (SML)} \tab Leave-one-out algorithm using exponential weighting. Proposed by \insertRef{zhang2024winners}{argminCS}. \cr
  \tab \cr
  \code{argmin.LOO (HML)} \tab A variant of SML that uses hard argmin instead of exponential weighting. Not recommended. \cr
  \tab \cr
  \code{nonsplit (NS)} \tab A variant of SML without data splitting. Requires a fixed lambda value as an additional argument. Not recommended\cr
  \tab \cr
  \code{Bonferroni (MT)} \tab Multiple testing using Bonferroni correction. \cr
  \tab \cr
  \code{Gupta (GTA)} \tab The method proposed by \insertCite{gupta.1965;textual}{argminCS}.
  Requires independence and the same population standard deviation for all dimensions. \cr
  \tab \cr
  \code{Futschik (FCHK)} \tab A two-step method from \insertCite{futschik.1995;textual}{argminCS}.
  Requires independence and the same population standard deviation for all dimensions. \cr
}
}
\examples{
r <- 4
n <- 200
mu <- (1:20)/20
cov <- diag(length(mu))
set.seed(108)
data <- MASS::mvrnorm(n, mu, cov)
sample.mean <- colMeans(data)

## softmin.LOO
CS.argmin(data)

## use seed
CS.argmin(data, seed=13)

## argmin.LOO
CS.argmin(data, method='HML')

## nonsplit
CS.argmin(data, method='NS', lambda=sqrt(n)/2.5)

## Bonferroni (choose t test because of normal data)
CS.argmin(data, method='MT', test='t')

## Gupta
CS.argmin(data, method='GTA')

## Futschik two-step method
# default alpha.1, alpha.2
CS.argmin(data, method='FCHK')

alpha.1 <- 0.0005
alpha.2 <- 1 - (0.95/(1 - alpha.1))
CS.argmin(data, method='FCHK', alpha.1=0.0005, alpha.2=alpha.2)

}
\references{
{
 \insertRef{zhang2024winners}{argminCS}

 \insertRef{cck.many.moments}{argminCS}

 \insertRef{gupta.1965}{argminCS}

 \insertRef{futschik.1995}{argminCS}
}
}
