% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fittestArimaKF.r
\name{fittestArimaKF}
\alias{fittestArimaKF}
\title{Automatic ARIMA fitting and prediction with Kalman filter}
\usage{
fittestArimaKF(
  timeseries,
  timeseries.test = NULL,
  h = NULL,
  na.action = stats::na.omit,
  level = 0.9,
  filtered = TRUE,
  initQ = NULL,
  rank.by = c("MSE", "NMSE", "MAPE", "sMAPE", "MaxError", "AIC", "AICc", "BIC", "logLik",
    "errors", "fitness"),
  ...
)
}
\arguments{
\item{timeseries}{A vector or univariate time series which contains the
values used for fitting an ARIMA model with Kalman filter.}

\item{timeseries.test}{A vector or univariate time series containing a
continuation for \code{timeseries} with actual values. It is used as a
testing set and base for calculation of prediction error measures. Ignored
if \code{NULL}.}

\item{h}{Number of consecutive values of the time series to be predicted. If
\code{h} is \code{NULL}, the number of consecutive values to be predicted is
assumed to be equal to the length of \code{timeseries.test}. Required when
\code{timeseries.test} is \code{NULL}.}

\item{na.action}{A function for treating missing values in \code{timeseries}
and \code{timeseries.test}. The default function is \code{\link[stats]{na.omit}},
which omits any missing values found in \code{timeseries} or
\code{timeseries.test}.}

\item{level}{Confidence level for prediction intervals. See the
\code{\link[KFAS]{predict.SSModel}} function in the \code{KFAS} package.}

\item{filtered}{If \code{filtered} is \code{TRUE}, Kalman filtered time
series observations are used for prediction, otherwise, Kalman smoothed
observations are used for prediction.}

\item{initQ}{Numeric argument regarding the initial value for the covariance
of disturbances parameter to be optimized over. The initial value to be
optimized is set to \code{exp(initQ)}. See the \code{Q} argument of the
\code{\link[KFAS]{SSMarima}} function in the \code{KFAS} package and the examples
in \code{\link[KFAS]{KFAS}}. If \code{NULL}, \code{initQ} is automatically set.
See 'Details'.}

\item{rank.by}{Character string. Criteria used for ranking candidate models
generated using different options of values for \code{initQ}. Only used if
\code{initQ} is \code{NULL}. Ignored otherwise. See 'Details'.}

\item{...}{Additional arguments passed to the \code{\link[forecast]{auto.arima}}
modelling function.}
}
\value{
A list with components: \item{model}{An object of class "SSModel"
containing the best evaluated ARIMA model fitted with Kalman Filter.}
\item{initQ}{The initQ argument provided (or automatically selected) for
optimization of the best evaluated ARIMA model fitted with Kalman Filter.}
\item{AICc}{Numeric value of the computed AICc criterion of the best
evaluated model.} \item{AIC}{Numeric value of the computed AIC criterion of
the best evaluated model.} \item{BIC}{Numeric value of the computed BIC
criterion of the best evaluated model.} \item{logLik}{Numeric value of the
computed log-likelihood of the best evaluated model.} \item{pred}{A list
with the components \code{mean}, \code{lower} and \code{upper}, containing
the predictions of the best evaluated model and the lower and upper limits
for prediction intervals, respectively. All components are time series. See
\code{\link[KFAS]{predict.SSModel}}.} \item{MSE}{Numeric value of the resulting
MSE error of prediction. Require \code{timeseries.test}.}
\item{NMSE}{Numeric value of the resulting NMSE error of prediction. Require
\code{timeseries.test}.} \item{MAPE}{Numeric value of the resulting MAPE
error of prediction. Require \code{timeseries.test}.} \item{sMAPE}{Numeric
value of the resulting sMAPE error of prediction. Require
\code{timeseries.test}.} \item{MaxError}{Numeric value of the maximal error
of prediction. Require \code{timeseries.test}.} \item{rank.val}{Data.frame
with the fitness or prediction accuracy criteria computed for all candidate
ARIMA with Kalman filter models ranked by \code{rank.by}. It has the
attribute \code{"ranked.models"}, which is a list of objects of class
"SSModel" containing all the candidate ARIMA models fitted with Kalman
Filter, also ranked by \code{rank.by}. Only provided if \code{initQ} was
automatically selected.} \item{rank.by}{Ranking criteria used for ranking
candidate models and producing \code{rank.val}.}
}
\description{
The function predicts and returns the next n consecutive values of a
univariate time series using the best evaluated ARIMA model automatically
fitted with Kalman filter. It also evaluates the fitness of the produced
model, using AICc, AIC, BIC and logLik criteria, and its prediction
accuracy, using the MSE, NMSE, MAPE, sMAPE and maximal error accuracy
measures.
}
\details{
A best ARIMA model is automatically fitted by the \code{\link[forecast]{auto.arima}}
function in the \code{forecast} package. The coefficients of this model are
then used as initial parameters for optimization of a state space model
(\code{\link[KFAS]{SSModel}}) using the Kalman filter and functions of the
\code{KFAS} package (see \code{\link[KFAS]{SSMarima}} and
\code{\link[KFAS]{artransform}}).

If \code{initQ} is \code{NULL}, it is automatically set as either
\code{log(var(timeseries))} or \code{0}. For that, a set of candidate ARIMA
state space models is generated by different initial parameterization of
\code{initQ} during the model optimization process. The value option which
generates the best ranked candidate ARIMA model acoording to the criteria in
\code{rank.by} is selected.

The ranking criteria in \code{rank.by} may be set as a prediction error
measure (such as \code{\link{MSE}}, \code{\link{NMSE}}, \code{\link{MAPE}},
\code{\link{sMAPE}} or \code{\link{MAXError}}), or as a fitness criteria
(such as \code{\link{AIC}}, \code{\link[MuMIn]{AICc}}, \code{\link{BIC}} or
\code{\link{logLik}}). In the former case, the candidate models are used for
time series prediction and the error measures are calculated by means of a
cross-validation process. In the latter case, the candidate models are
fitted and fitness criteria are calculated based on all observations in
\code{timeseries}.

If \code{rank.by} is set as \code{"errors"} or \code{"fitness"}, the
candidate models are ranked by all the mentioned prediction error measures
or fitness criteria, respectively. The wheight of the ranking criteria is
equally distributed. In this case, a \code{rank.position.sum} criterion is
produced for ranking the candidate models. The \code{rank.position.sum}
criterion is calculated as the sum of the rank positions of a model (1 = 1st
position = better ranked model, 2 = 2nd position, etc.) on each calculated
ranking criteria.
}
\examples{

\donttest{
data(CATS,CATS.cont)
fArimaKF <- fittestArimaKF(CATS[,2],CATS.cont[,2])
#predicted values
pred <- fArimaKF$pred

#extracting Kalman filtered and smoothed time series from the best fitted model
fs <- KFAS::KFS(fArimaKF$model,filtering=c("state","mean"),smoothing=c("state","mean"))
f <- fitted(fs, filtered = TRUE) #Kalman filtered time  series
s <- fitted(fs) #Kalman smoothed time  series
#plotting the time series data
plot(c(CATS[,2],CATS.cont[,2]),type='o',lwd=2,xlim=c(960,1000),ylim=c(200,600),
 xlab="Time",ylab="ARIMAKF")
#plotting the Kalman filtered time series
lines(f,col='red',lty=2,lwd=2)
#plotting the Kalman smoothed time series
lines(s,col='green',lty=2,lwd=2)
#plotting predicted values
lines(ts(pred$mean,start=981),lwd=2,col='blue')
#plotting prediction intervals
lines(ts(pred$upper,start=981),lwd=2,col='light blue')
lines(ts(pred$lower,start=981),lwd=2,col='light blue')
}

}
\references{
R.J. Hyndman and G. Athanasopoulos, 2013, Forecasting:
principles and practice. OTexts.

R.H. Shumway and D.S. Stoffer, 2010, Time Series Analysis and Its
Applications: With R Examples. 3rd ed. 2011 edition ed. New York, Springer.
}
\seealso{
\code{\link{fittestArima}}, \code{\link{fittestLM}},
\code{\link{marimapred}}
}
\author{
Rebecca Pontes Salles
}
\keyword{ARIMA}
\keyword{Kalman}
\keyword{adjustment}
\keyword{automatic}
\keyword{criterion}
\keyword{errors}
\keyword{evaluation}
\keyword{filter}
\keyword{fitting}
\keyword{prediction}
