#
# plot_final_year_migration_data.R
#
#' Plot daily data on migration fluxes by actively mobile guilds during the final year of a run.
#'
#' Generate a multi-panel set of one-year time series plots of of the mass fluxes between inshore and offshore zones due to migration by actively mobile guilds in the 
#' ecology model: all three fish guilds, birds, pinnipeds and cetaceans. This function creates the default plot from a single model run.
#'
#' The units of data shown in each panel are mMN/d in the model domain as a whole, assumed to be 1m2.
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2e_read() function.
#' @param results List object of single-run model output generated by running the function e2e_run() (default = NULL).
#'
#' @return graphical display in a new graphics window
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_final_year_migration_data <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	build		<- elt(results, "build")
	run		<- elt(build, "run")
	nyears		<- elt(run, "nyears")
	ndays		<- elt(run, "ndays")

	data		<- elt(model, "data")
	physical.parms	<- elt(data, "physical.parameters")
	x_shallowprop	<- elt(physical.parms, "x_shallowprop")

	aggregates	<- elt(results, "aggregates")


	cumpelfish_o_i <- aggregates$NET_activemigpelfish_o_i[((nyears-1)*360+1):ndays]
	cummigfish_o_i <- aggregates$NET_activemigmigfish_o_i[((nyears-1)*360+1):ndays]
	cumdemfish_o_i <- aggregates$NET_activemigdemfish_o_i[((nyears-1)*360+1):ndays]
	cumbird_o_i    <- aggregates$NET_activemigbird_o_i[((nyears-1)*360+1):ndays]
	cumseal_o_i    <- aggregates$NET_activemigseal_o_i[((nyears-1)*360+1):ndays]
	cumceta_o_i    <- aggregates$NET_activemigceta_o_i[((nyears-1)*360+1):ndays]


#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Migration flux plotting function

fyplot1mig<-function(tspmain,axtitle,tspvar1){
par(mar=c(3.5,5.4,2.1,0.7))
tsyears<-length(tspvar1)
tseq<-seq(0,360,by=1)
ymax<-(max(tspvar1))
ymin<-(min(tspvar1))
  if(ymax>0) plmax<-1.05*ymax
  if(ymax<0) plmax<-0
  if(ymax==0) plmax<-0
  if(ymin>0) plmin<-0
  if(ymin<0) plmin<-1.05*ymin
  if(ymin==0) plmin<-0
plot(tseq,tspvar1,type="l",yaxt="n",xlim=c(0,360),ylim=c(plmin,plmax),xaxt="n",ann=FALSE,main=tspmain)
axis(side=1,at=c(0,60,120,180,240,300,360),las=1,cex.axis=1.0)
axis(side=2,las=1,cex.axis=1.0)
mtext("Days",cex=0.8,side=1,line=2)
mtext(axtitle,cex=0.8,side=2,line=3.95)
mtext(tspmain,cex=0.85,side=3,line=0.5)
abline(a=0,b=0)
}

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# Function to convert cumulative flux into daily increments

diconvert<-function(v2proc){
tempset<-v2proc
tempset[362] <-v2proc[361] + (v2proc[361] - v2proc[360])
converted <- tempset[2:362] - v2proc
return(converted)
}

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

# The flux dat are cumulative so first thing to do is convert to daily incrtements

	pelfish_o_i <- diconvert(cumpelfish_o_i)
	migfish_o_i <- diconvert(cummigfish_o_i)
	demfish_o_i <- diconvert(cumdemfish_o_i)
	bird_o_i    <- diconvert(cumbird_o_i)
	seal_o_i    <- diconvert(cumseal_o_i)
	ceta_o_i    <- diconvert(cumceta_o_i)


unitsize<-0.55
unitline<- -0.2
unitadj<- -1.25
par(mfrow=c(2,3))

l1<-pelfish_o_i
fyplot1mig("Planktivorous fish","Active migration flux",l1)
legend("topright",  bty="n","Offshore to inshore",cex=0.9)
legend("bottomright",  bty="n","Inshore to offshore",cex=0.9)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

l1<-migfish_o_i
fyplot1mig("Migratory fish","Active migration flux",l1)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

l1<-demfish_o_i
fyplot1mig("Demersal fish","Active migration flux",l1)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

l1<-bird_o_i
fyplot1mig("Birds","Active migration flux",l1)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

l1<-seal_o_i
fyplot1mig("Pinnipeds","Active migration flux",l1)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

l1<-ceta_o_i
fyplot1mig("Cetaceans","Active migration flux",l1)
#		mtext(bquote("mMN.m"^-2 ~ ".d"^-1),cex=unitsize,side=3,line=unitline,adj=unitadj)

}

