#
# plot_df_yield_curve.R
#
#' Plot a fishery yield curve data for demersal fish.
#'
#' Plot demersal fish yield curve data generated by the function generate_df_yield_curve_data(). 
#' 
#' In the function generate_df_yield_curve_data(), the baseline for the sequence of runs (harvest ratio multiplier = 1.0) is a model name and variant as loaded my the e2e_read() function.
#' The function generates a set of biomass, landings and discards data for multiples of demersal fish harvest ratios relative to this baseline. This is done for a given value of
#' planktiviorous harvest ratio (also a multiple of the the baseline). Data returned from the generate_df_yield_curve_data() are in form of a dataframe, and optionally saved to a csv file.
#'
#' This function plots two graphs - the annual average demersal fish biomass (mMN/m2) as a function of demersal fish harvest ratio multiplier, and the yield curve, ie the annual catch (and discards) (mMN/m2/y) as functions of the multiplier.
#'
#' The yield curve represents the catch that would be generated from the stationary state of the model attained with long-term repeating annual cycles of all driving data.
#'
#' Arguments for this function permit the input data to be drawn from an existing data object generated by the function generate_df_yield_curve_data(), a previously generted csv file, or example data
#' provided with the package for versions of the internal North Sea models.
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2e_read() function.
#' @param use.saved Logical. If TRUE use pre-computed data saved from a prior run of the function rather than user-generated dataframe object (default=FALSE)
#' @param use.example Logical. If TRUE use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE)
#' @param results Dataframe generated by the function generate_df_yield_curve_data() (default = NULL)
#' @param title Free text (enclosed in "") to be added as a header for the plot (default = "")
#'
#' @return Dataframe of results from which the plot is created, graphical display in a new graphics window.
#'
#' @importFrom graphics text
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_df_yield_curve <- function(model,use.saved=FALSE, use.example=FALSE, results=NULL ,title="" ) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

if(use.saved==TRUE & use.example==TRUE){
        stop("use.saved and use.example cannot both be TRUE ! \n")
}

if(use.saved==TRUE & is.data.frame(results)==TRUE){
        stop("use.saved is TRUE but a dataframe object has also been specified ! \n")
}

if(use.example==TRUE & is.data.frame(results)==TRUE){
        stop("use.example is TRUE but a dataframe object has also been specified ! \n")
}

if(use.saved==FALSE & use.example==FALSE & is.data.frame(results)==FALSE){
        stop("no source of data has been specified ! \n")
}


if(use.example==TRUE){
	model.name 	<- elt(model, "setup", "model.name")
	model.variant 	<- elt(model, "setup", "model.variant")
	datastore <- get.example.results(model.name, model.variant, "Yield_curve_data_DFHRmult")
	if(title=="") title<-paste("Example data for ",model.name," ",model.variant," model")
}

if(use.saved==TRUE) {
	resultsdir	<- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")
	datafile	<- csvname(resultsdir, "Yield_curve_data_DFHRmult", model.ident)
	print(paste("Using data held in a file ",datafile," from a past model run"))
	check.exists(datafile)
	datastore<-readcsv(datafile)
}

if(use.saved==FALSE & use.example==FALSE & is.data.frame(results)==TRUE) datastore<-results

#--------------------------------------------------------

#Plot results of DFHR set
par(mfrow=c(2,1))
par(mar=c(3.2,5,2,0.8))

ym<-1.1*max(datastore$DemFishbiom)
plot(datastore$DemFishHRmult,datastore$DemFishbiom,ylim=c(0,ym),type="l",lwd=3,yaxt="n",xaxt="n",ann=FALSE)
abline(v=1,lty="dashed")
	axis(side=1,las=1,cex.axis=0.9)
	axis(side=2,las=1,cex.axis=0.9)
	mtext("Demersal fish harvest ratio multiplier",cex=1,side=1,line=2)
	mtext("Demersal fish biomass",cex=1,side=2,line=3)
	mtext(title,cex=1.2,side=3,line=0.7)
	mtext(bquote("mMN.m"^-2),cex=0.7,side=3,line=-0.05,adj=-0.18)

ym<-1.1*max(datastore$DemFishland+datastore$DemFishdisc)
plot(datastore$DemFishHRmult,datastore$DemFishland,ylim=c(0,ym),type="l",yaxt="n",xaxt="n",ann=FALSE)
lines(datastore$DemFishHRmult,datastore$DemFishland+datastore$DemFishdisc,type="l",lwd=3)
abline(v=1,lty="dashed")
	axis(side=1,las=1,cex.axis=0.9)
	axis(side=2,las=1,cex.axis=0.9)
	mtext("Demersal fish harvest ratio multiplier",cex=1,side=1,line=2)
	mtext("Demersal fish catch",cex=1,side=2,line=3)
        mtext(bquote("mMN.m"^-2 ~ ".y"^-1),cex=0.7,side=3,line=-0.05,adj=-0.18)
legend("topright",  bg="transparent", c("Catch","Landings"), lty=c("solid","solid"),lwd=c(3,1),cex=0.8)

#---------------------------

return(datastore)

}

