#' @importFrom Rssa ssa rforecast reconstruct wcor
#' @importFrom utils tail head
#' @importFrom stats ts residuals
#' @export
#'

SSAfcast <- function(data, L = 12, corr_thr = 0.97, horizon = 12) {
  # Training and test split
  data_trn <- head(data, length(data) - horizon)
  data_test <- tail(data, horizon)

  # First SSA decomposition (trend extraction)
  s1 <- Rssa::ssa(data_trn, L)
  res <- Rssa::reconstruct(s1, groups = list(1))
  trend <- res$F1
  res.trend <- data_trn - trend

  # Second SSA decomposition (residual analysis)
  L1 <- L * floor((length(data_trn) / 2) / L)
  s2 <- Rssa::ssa(res.trend, L1)
  wtcor <- Rssa::wcor(s2)

  # Select highly correlated component pairs
  num_components <- nrow(wtcor)
  high_correlation_pairs <- list()
  for (i in 1:(num_components - 1)) {
    for (j in (i + 1):num_components) {
      corr_ij <- wtcor[i, j]
      if (corr_ij > corr_thr) {
        high_correlation_pairs <- append(high_correlation_pairs, list(c(i, j, corr_ij)))
      }
    }
  }

  # Reconstruct and forecast selected groups
  selected_groups <- list()
  forecasted_components <- list()
  for (pair in high_correlation_pairs) {
    group <- c(pair[1], pair[2])
    selected_groups <- append(selected_groups, list(group))
    forecasted <- Rssa::rforecast(s2, groups = list(group), len = horizon, only.new = TRUE)
    forecasted_components[[paste("Forecasted", paste(group, collapse = "-"))]] <- forecasted
  }

  # Sum all forecasted components
  sum_forecast <- Reduce("+", forecasted_components)

  # Add trend forecast
  frssa1 <- Rssa::rforecast(s1, groups = list(1), len = horizon, only.new = TRUE)
  final_forecast <- frssa1 + sum_forecast

  # Accuracy metrics
  RMSE_SSA <- sqrt(mean((data_test - final_forecast)^2))
  MAPE_SSA <- mean(abs((data_test - final_forecast) / data_test)) * 100
  MAE_SSA <- mean(abs(data_test - final_forecast))

  return(list(
    Final_forecast = final_forecast,
    Data_test = data_test,
    RMSE_SSA = RMSE_SSA,
    MAPE_SSA = MAPE_SSA,
    MAE_SSA = MAE_SSA
  ))
}


