## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 4
)

## ----eval=FALSE---------------------------------------------------------------
# # From CRAN (when available)
# install.packages("RegimeChange")
# 
# # From GitHub
# # devtools::install_github("yourusername/RegimeChange")

## ----message=FALSE, warning=FALSE---------------------------------------------
library(RegimeChange)

# Generate example data with a changepoint at t=100
set.seed(123)
data <- c(rnorm(100, mean = 0, sd = 1), rnorm(100, mean = 3, sd = 1))

# Detect changepoints using PELT (default)
result <- detect_regimes(data)

# Print results
print(result)

## -----------------------------------------------------------------------------
# Basic plot
plot(result)

# Segment-colored plot
plot(result, type = "segments")

## ----eval=FALSE---------------------------------------------------------------
# # PELT - Pruned Exact Linear Time (default)
# result_pelt <- detect_regimes(data, method = "pelt")
# 
# # Binary Segmentation
# result_binseg <- detect_regimes(data, method = "binseg")
# 
# # Wild Binary Segmentation
# result_wbs <- detect_regimes(data, method = "wbs")
# 
# # CUSUM
# result_cusum <- detect_regimes(data, method = "cusum")

## ----eval=FALSE---------------------------------------------------------------
# # BOCPD - Bayesian Online Changepoint Detection
# result_bocpd <- detect_regimes(data, method = "bocpd")
# 
# # Shiryaev-Roberts
# result_sr <- detect_regimes(data, method = "shiryaev", mu0 = 0, mu1 = 3, sigma = 1)

## ----eval=FALSE---------------------------------------------------------------
# # Mean changes (default)
# detect_regimes(data, type = "mean")
# 
# # Variance changes
# detect_regimes(data, type = "variance")
# 
# # Both mean and variance
# detect_regimes(data, type = "both")
# 
# # Trend changes
# detect_regimes(data, type = "trend")

## ----eval=FALSE---------------------------------------------------------------
# # Normal-Gamma prior for unknown mean and variance
# prior <- normal_gamma(mu0 = 0, kappa0 = 1, alpha0 = 1, beta0 = 1)
# result <- detect_regimes(data, method = "bocpd", prior = prior)
# 
# # Geometric hazard prior
# hazard <- geometric_hazard(lambda = 0.01)
# result <- detect_regimes(data, method = "bocpd", hazard = hazard)

## -----------------------------------------------------------------------------
# True changepoint is at position 100
evaluation <- evaluate(result, true_changepoints = 100, tolerance = 5)
print(evaluation)

## ----eval=FALSE---------------------------------------------------------------
# comparison <- compare_methods(
#   data = data,
#   methods = c("pelt", "bocpd", "binseg"),
#   true_changepoints = 100
# )
# print(comparison)

## ----eval=FALSE---------------------------------------------------------------
# # Create online detector
# detector <- regime_detector(method = "bocpd", prior = normal_gamma())
# 
# # Process observations one at a time
# for (x in data) {
#   result <- update(detector, x)
#   detector <- result$detector
# 
#   if (result$alarm) {
#     cat("Changepoint detected at observation", length(detector$history), "\n")
#   }
# }

## ----eval=FALSE---------------------------------------------------------------
# result <- detect_regimes(data, uncertainty = TRUE, bootstrap_reps = 100)
# print(result$confidence_intervals)

