#' @title Shape difference between 3d meshes
#' @description The function reconstructs two specimens' meshes (\code{x})
#'   by using their superimposed configurations (from within \code{pca}),
#'   calculates the shape difference between them, and plots such differences
#'   on provided meshes (\code{refsur}).
#' @usage shapeDiff(x,pca,refsur,refmat,mshape_sur = NULL,
#'   pal=NULL,NAcol="gray90",show.names=TRUE)
#' @param x a vector of specimens pair.
#' @param pca the result of a relative warp analysis. Classes \code{relwarps} and
#'   \code{nosymproc} are both accepted.
#' @param refsur a list of two \code{mesh3d} objects to be provided in the same order as \code{x}.
#' @param refmat a list of two landmark sets related to \code{refsur} to be
#'   provided in the same order as \code{x}.
#' @param mshape_sur a \code{mesh3d} object used as a reference for mesh reconstruction.
#'   The vertices of \code{mshape_sur} must be the consensus configuration. If \code{NULL},
#'   it is automatically generated by applying \code{\link[Rvcg]{vcgBallPivoting}}
#'   on the consensus configuration derived from \code{pca}.
#' @param pal a vector of colors to be passed to \code{\link[grDevices]{colorRampPalette}}.
#' @param NAcol the color associated to \code{refsur} vertices falling outside the
#'   range of \code{refmat} (not involved in interpolation).
#' @param show.names logical: if \code{TRUE}, the names of the specimens as in \code{x}
#'   are displayed in the 3d plot.
#' @return Two \code{mesh3d} objects colored according to shape differences.
#'   Additionally, the function returns 3d plots of the meshes.
#' @export
#' @author Marina Melchionna, Silvia Castiglione
#' @importFrom Morpho closemeshKD
#' @examples
#'   \donttest{
#'   da<-"https://github.com/pasraia/RRmorph_example_data/raw/refs/heads/main/RRmorphdata.rda"
#'   download.file(url=da,destfile = paste0(tempdir(),"/RRmorphdata.rda"))
#'   load(paste0(tempdir(),"/RRmorphdata.rda"))
#'
#'   require(Morpho)
#'
#'   pca<-procSym(endo.set)
#'   ldm_homo<-endo.set[,,"Homo_sapiens"]
#'   sur_homo<-endo.sur[["Homo_sapiens"]]
#'   ldm_macaca<-endo.set[,,"Macaca_fuscata"]
#'   sur_macaca<-endo.sur[["Macaca_fuscata"]]
#'
#'   diffs<-RRmorph::shapeDiff(x=c("Homo_sapiens","Macaca_fuscata"),
#'                    pca = pca,refsur = list(sur_homo,sur_macaca),
#'                    refmat = list(ldm_homo,ldm_macaca))
#'   }

shapeDiff<-function(x,pca,refsur,refmat,mshape_sur=NULL,
                    pal=NULL,NAcol="gray90",show.names=TRUE){

  if(!is.list(refsur))   stop("Please, provide refsur as a list")
  if(!is.list(refmat))   stop("Please, provide refmat as a list")

  if(inherits(pca,"relwarps")){
    pca$bescores->PCscores
    pca$mshape->mshape
    pca$bePCs->pcs
    pca$rotated->rot
  }

  if(inherits(pca,"nosymproc")){
    pca$PCscores->PCscores
    pca$mshape->mshape
    pca$PCs->pcs
    pca$rotated->rot
  }

  if(is.null(pal)) pal<-c("#0a1045","#00c2d1","white","gold2","orangered")
  # colmap_tot <- colorRampPalette(pal)

  refsur[[2]]<-rotmesh.onto(refsur[[2]],refmat[[2]],refmat[[1]],scale = FALSE)$mesh
  refmat[[2]]<-rotmesh.onto(refsur[[2]],refmat[[2]],refmat[[1]],scale = FALSE)$yrot

  sur<-lapply(x, function(i) vcgBallPivoting(rot[,,i],radius = 0))
  if(is.null(mshape_sur)) mshape_sur<-vcgBallPivoting(mshape,radius = 0)

  sur[[1]]$it <- sur[[2]]$it <- mshape_sur$it

  diffs<-list()
  diffs[[1]]<-closemeshKD(sur[[2]], sur[[1]],sign = TRUE)$quality[,1]
  diffs[[2]]<-closemeshKD(sur[[1]], sur[[2]],sign = TRUE)$quality[,1]

  values<-mapply(a=sur,b=refsur,c=refmat,d=diffs, function(a,b,c,d)
    interpolMesh(a,d,b,c,element="point"), SIMPLIFY = FALSE)

  open3d()
  mfrow3d(1,2,sharedMouse = TRUE)

  meshes<-list()

  for(w in 1:length(values)){
    meshes[[w]]<-col2mesh(refsur[[w]],values[[w]],pal,NAcol=NAcol)
    plotLegend(meshes[[w]],values[[w]],main=x[w])

    if(w==2) next3d()
    shade3d(meshes[[w]], specular = "black")
    if (show.names) mtext3d(text = paste(x[w]), font = 2,edge = "x", line = 3)

  }

  return(meshes)
}
