###
### Print Method for the RCT2 Package
### 
###



#' Print Method for the RCT2 Package
#'
#' 
#' This function prints a nicely formatted summary of the three functions in the RCT2 package.
#'
#' 
#' For the details of the method implemented by this function, see the
#' references.
#' 
#' @param x A list object generated by running one of the analyses on a data set.
#' @param ... ignored
#' @return \code{NULL}
#' 
#' @author Kosuke Imai, Department of Statistics, Harvard University
#' \email{imai@harvard.edu}, \url{https://imai.fas.harvard.edu/};
#' Zhichao Jiang, School of Public Health and Health Sciences, University of Massachusetts Amherst
#' \email{zhichaojiang@umass.edu};
#' Karissa Huang, Department of Statistics, Harvard College
#' \email{krhuang@college.harvard.edu}
#' @references Kosuke Imai, Zhichao Jiang and Anup Malani (2018).
#' \dQuote{Causal Inference with Interference and Noncompliance in the Two-Stage Randomized Experiments}, \emph{Technical Report}. Department of Politics, Princeton
#' University.
#' @keywords two-stage randomized experiments
#' @export




#' @method print regression
print.regression <- function(x, ...){
  CIs <- rbind(x$CADE1.CI, x$CADE0.CI)
  CADEs <- rbind(x$CADE1, x$CADE0)
  name <- c("CADE1", "CADE0")
  variances <- c("Cluster Robust Variance", "HC2 Variance", "Cluster Robust HC2 Variance", "Individual Variance", "Proposed Variance")
  cluster_robust_variance <- c(x$var1.clu, x$var0.clu)
  HC2_variance <- c(x$var1.hc2, x$var0.hc2)
  cluster_robust_HC2_variance <- c(x$var1.clu.hc2, x$var0.clu.hc2)
  individual_variance <- c(x$var1.ind, x$var0.ind)
  proposed_variance <- c(x$var1.reg, x$var0.reg)
  variance_values <- as.data.frame(rbind(cluster_robust_variance, HC2_variance, cluster_robust_HC2_variance, individual_variance, proposed_variance))
  names(variance_values) <- c("var(CADE1)", "var(CADE0)")
  estimates <- as.data.frame(cbind(name, CADEs, CIs))
  rownames(estimates) <- NULL
  colnames(estimates) <- c("name", "estimate", "left CI", "right CI")
  return(list(estimates, variance_values))
}

#' @export
#' @method print random
print.random <- function(x, ...){
  variance <- c(x$var.CADE, x$var.CASE, x$var.DEY, x$var.DED, x$var.SEY, x$var.SED)
  stds <- sqrt(variance)
  estimate <- c(x$CADE, x$CASE, x$DEY, x$DED, x$SEY, x$SED)
  
  
  name <- c("CADE", "CASE", "DEY", "DED", "SEY", "SED")
  names <- rep(name, each = length(x$CADE))
  reps <- seq(0, length(x$CASE)-1, 1)
  reps2 <- rep(reps, times = length(name))

  variance <- round(variance, 5)
  stds <- round(stds, 5)
  estimate <- round(estimate, 5)
  lcis <- c(x$lci.CADE, x$lci.CASE, x$lci.DEY, x$lci.SEY, x$lci.DED, x$lci.SED)
  rcis <- c(x$rci.CADE,x$rci.CASE, x$rci.DEY, x$rci.SEY, x$rci.DED, x$rci.SED)
 

  out <- as.data.frame(cbind(names, reps2, estimate, variance, stds, lcis, rcis))
  rownames(out) <- NULL
  return(out)
}




#' @export
#' @method print parametric
print.parametric <- function(x, ...){
  names <- c("ITT DE", "IV DE", "ITT SE", "IV SE")
  effects <- rbind(x$ITT.DE, x$IV.DE, x$ITT.SE, x$IV.SE)
  CIs <- rbind(x$ITT.DE.CI, x$ITT.SE.CI, x$IV.DE.CI, x$IV.SE.CI)
  init <- as.data.frame(cbind(names, effects, CIs))
  colnames(init) <- c("Method", "Treatment", "Control", "Treatment CI", "Control CI")
  tstat_pvals <- as.data.frame(cbind(x$ITT.pvals, x$ITT.tstat, x$IV.pvals, x$IV.tstat))
  colnames(tstat_pvals) <- c("ITT pvalues", "ITT tstat", "IV pvalues", "IV tstat")
  out <- list(init, tstat_pvals)
  return(out)
}

#' @export
#' @method print sample
print.sample <- function(x, ...){
  n <- length(x)
  index <- seq(1, n, 1)
  out <- rbind(as.integer(index), x)
  rownames(out) <- c("Assignment Mechanism", "Number of Clusters")
  colnames(out) <- NULL
  return(out)
}

#' @export
#' @method print CalAPO

print.CalAPO <- function(x, ...){
  Y.hat <- as.data.frame(x$Y.hat)
  m <- nrow(Y.hat)/2
  colnames(Y.hat) <-  "Potential Outcome Estimates"
  rownames(Y.hat) <- paste(c("treated group", "control group"), rep(1:m, each = 2), "estimate")
  
  ADE <- as.data.frame(x$ADE.est)
  colnames(ADE) <- "Average Direct Effect"
  rownames(ADE) <- paste("assignment group", rep(1:m))
  
  ASE <- as.data.frame(x$ASE.est)
  colnames(ASE) <- "Average Spillover Effect"
  rownames(ASE) <- paste(c(rep("treatment group under assignments", 2), rep("control group under assignments", 2)), seq(1, m-1), seq(2, m))
  
  MDE <- as.data.frame(x$MDE.est)
  colnames(MDE) <- "Marginal Direct Effect"
  
  return(list(Y.hat, Y.covariance = x$cov.hat, ADE, ADE.covariance = x$var.hat.ADE, ASE, ASE.covariance = x$var.hat.ASE, MDE, MDE.covariance = x$var.hat.MDE))
}







