#' Perform QA/QC on spectral data while preserving original column names
#'
#' @param file_path Path to the input file
#' @param output_path Path to save the cleaned data
#' @param handle_missing Method to handle missing values ('impute', 'remove', or 'NA')
#' @param handle_outliers Method to handle outliers ('impute', 'remove', or 'NA')
#' @param group_by_col Column name for grouping
#' @return A list with cleaned data and a summary table
#' @export
#' @examples
#' library(openxlsx)
#' # Create mock raw data
#' raw_data <- data.frame(
#'   treatment = sample(0:1, 100, replace = TRUE),
#'   var1 = rnorm(100),
#'   var2 = rnorm(100),
#'   var3 = rnorm(100)
#' )
#'
#' # Save mock data to a temporary file
#' raw_data_file <- tempfile(fileext = ".xlsx")
#' output_file <- tempfile(fileext = ".xlsx")
#'
#' write.xlsx(raw_data, raw_data_file)
#'
#' # Run QA/QC with missing values imputed and outliers removed
#' cleaned_result <- qaqcs(
#'   file_path = raw_data_file,
#'   output_path = output_file,
#'   handle_missing = "impute",
#'   handle_outliers = "remove",
#'   group_by_col = "treatment"
#' )
#' head(cleaned_result$cleaned_data)
qaqcs <- function(file_path, output_path, handle_missing = "NA", handle_outliers = "NA", group_by_col = "treatment") {
  
  # Step 1: Read the dataset while preserving original column names
  data <- read_xlsx(file_path)
  
  # Save original column names
  original_colnames <- colnames(data)
  
  # Step 2: Detect Missing Values
  missing_report <- data %>%
    summarise(across(everything(), ~sum(is.na(.)))) %>%
    pivot_longer(cols = everything(), names_to = "Variable", values_to = "MissingCount")
  
  # Step 3: Detect Outliers using IQR Method
  detect_outliers_group <- function(df, group_by_col) {
    df <- df %>%
      group_by(across(all_of(group_by_col))) %>%
      mutate(across(where(is.numeric), ~ {
        Q1 <- quantile(., 0.25, na.rm = TRUE)
        Q3 <- quantile(., 0.75, na.rm = TRUE)
        IQR <- Q3 - Q1
        is_outlier <- (. < (Q1 - 1.5 * IQR)) | (. > (Q3 + 1.5 * IQR))
        ifelse(is_outlier, TRUE, FALSE)
      }, .names = "Outlier_{.col}")) %>%
      ungroup()
    return(df)
  }
  
  # Add outlier flags
  data_with_outliers <- detect_outliers_group(data, group_by_col)
  
  # Summarize Outliers
  outlier_report <- data_with_outliers %>%
    summarise(across(starts_with("Outlier_"), ~sum(. == TRUE, na.rm = TRUE))) %>%
    pivot_longer(cols = everything(), names_to = "Variable", values_to = "OutlierCount") %>%
    mutate(Variable = gsub("Outlier_", "", Variable))
  
  # Merge Missing and Outlier Reports
  summary_table <- left_join(missing_report, outlier_report, by = "Variable") %>%
    mutate(MissingCount = coalesce(MissingCount, 0),
           OutlierCount = coalesce(OutlierCount, 0))
  
  # Step 4: Handle Missing Values
  if (handle_missing == "impute") {
    data <- data %>%
      group_by(across(all_of(group_by_col))) %>%
      mutate(across(where(is.numeric), ~ifelse(is.na(.), mean(., na.rm = TRUE), .))) %>%
      ungroup()
  } else if (handle_missing == "remove") {
    data <- data %>%
      filter(!if_any(everything(), is.na))
  }
  
  # Step 5: Handle Outliers
  if (handle_outliers == "impute") {
    data <- data_with_outliers %>%
      group_by(across(all_of(group_by_col))) %>%
      mutate(across(where(is.numeric), ~ifelse(get(paste0("Outlier_", cur_column())), mean(., na.rm = TRUE), .))) %>%
      ungroup() %>%
      select(-starts_with("Outlier_"))
  } else if (handle_outliers == "remove") {
    data <- data_with_outliers %>%
      filter(!if_any(starts_with("Outlier_"), ~. == TRUE)) %>%
      select(-starts_with("Outlier_"))
  } else if (handle_outliers == "NA") {
    data <- data_with_outliers %>%
      mutate(across(where(is.numeric), ~ifelse(get(paste0("Outlier_", cur_column())), NA, .))) %>%
      select(-starts_with("Outlier_"))
  }
  
  # Step 6: Restore original column names
  colnames(data) <- original_colnames
  
  # Step 7: Export data using writexl to preserve spaces
  write_xlsx(data, output_path)
  write_xlsx(summary_table, gsub("\\.xlsx", "_summary.xlsx", output_path))
  
  # Return cleaned data and summary table
  return(list(cleaned_data = data, summary_table = summary_table))
}
