% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/09_CALIBRATION.R
\name{rs.calibration}
\alias{rs.calibration}
\title{Calibration of the rating scale}
\usage{
rs.calibration(rs, dr, w, ct, min.pd, method)
}
\arguments{
\item{rs}{Rating scale that contain observed default rate and weights used for optimization.}

\item{dr}{Observed default rate per rating.}

\item{w}{Weights, usually number of observations (clients/accounts) per rating.}

\item{ct}{Value of central tendency to which calibration is performed.}

\item{min.pd}{Minimum probability of default (PD) per rating, as constrain for calibration process.}

\item{method}{Calibration method. Available options are \code{"scaling", "log.odds.a", "log.odds.ab"}.}
}
\value{
The command \code{rs.calibration} returns a list of two elements. The first element is
vector of calibrated PDs and the second one is dataframe of optimization parameters.
}
\description{
\code{rs.calibration} performs calibration of the observed default rates for a given rating scale.
}
\details{
Method \code{"scaling"} relies on linear rescaling of observed default rate. Rescaling factor is
calculated as a ratio between \code{ct} and observed portfolio default rate.
Method \code{"log.odds.a"} optimize intercept of logit transformation in a way that makes portfolio default
rate equal to selected central tendency (\code{ct}).
Method \code{"log.odds.ab"} optimize intercept and slope of logit transformation in a way that makes
portfolio default rate equal to selected central tendency (\code{ct}).
For respecting selected constrain of minimum PD (\code{min.pd}), two-stage iterative procedure is implemented.
Additional constrain of maximum PD (100\%) is also implemented.
}
\examples{
suppressMessages(library(PDtoolkit))
data(loans)
#estimate some dummy model
mod.frm <- `Creditability` ~ `Account Balance` + `Duration of Credit (month)` +
			`Age (years)`
lr.mod <- glm(mod.frm, family = "binomial", data = loans)
summary(lr.mod)$coefficients
#model predictions
loans$pred <-  unname(predict(lr.mod, type = "response", newdata = loans))
#scale probabilities
loans$score <- scaled.score(probs = loans$pred, score = 600, odd = 50/1, pdo = 20)
#group scores into rating
loans$rating <- sts.bin(x = round(loans$score), y = loans$Creditability, y.type = "bina")[[2]]
#create rating scale
rs <- loans \%>\%
group_by(rating) \%>\%
summarise(no = n(),
    nb = sum(Creditability),
    ng = sum(1 - Creditability)) \%>\%
mutate(dr = nb / no)
rs
#calcualte portfolio default rate
sum(rs$dr * rs$no / sum(rs$no))
#calibrate rating scale to central tendency of 27\% with minimum PD of 5\%
ct <- 0.33
min.pd <- 0.05
#scaling
pd.calib.s <- rs.calibration(rs = rs, 
			     dr = "dr", 
			     w = "no", 
			     ct = ct, 
			     min.pd = min.pd,
			     method = "scaling")
rs$pd.scaling <- pd.calib.s[[1]]
#log-odds a
pd.calib.a <- rs.calibration(rs = rs, 
			     dr = "dr", 
		 	     w = "no", 
			     ct = ct, 
			     min.pd = min.pd,
			     method = "log.odds.a")
rs$pd.log.a <- pd.calib.a[[1]]
#log-odds ab
pd.calib.ab <- rs.calibration(rs = rs, 
			      dr = "dr", 
		 	      w = "no", 
			      ct = ct, 
			      min.pd = min.pd,
			      method = "log.odds.ab")
rs$pd.log.ab <- pd.calib.ab[[1]]
#checks
rs
sum(rs$pd.scaling * rs$no / sum(rs$no))
sum(rs$pd.log.a * rs$no / sum(rs$no))
sum(rs$pd.log.ab * rs$no / sum(rs$no))
}
