#' @title EM algorithm for a univariate Gaussian mixture
#'
#' @description
#' Fits a univariate Gaussian mixture model using the Expectation-Maximization (EM)
#' algorithm. The function is intended as a lightweight fallback implementation
#' (e.g., when \pkg{mixtools} is unavailable or fails).
#'
#' @param X Numeric vector of observations.
#' @param K Integer. Number of mixture components.
#' @param max_iter Integer. Maximum number of EM iterations.
#' @param tol Positive numeric. Convergence tolerance for the absolute change in
#'   the log-likelihood.
#'
#' @return
#' A list with the following components:
#' \describe{
#'   \item{mu}{Numeric vector of estimated component means (length \code{K}).}
#'   \item{sigma}{Numeric vector of estimated component standard deviations (length \code{K}).}
#'   \item{pi}{Numeric vector of estimated mixing proportions (length \code{K}).}
#'   \item{num_iteraciones}{Number of iterations performed.}
#'   \item{posterior}{Matrix of posterior probabilities (responsibilities) with
#'   dimension \code{length(X)} by \code{K}.}
#' }
#'
#' @details
#' The algorithm is initialized using the k-means clustering procedure
#' and then alternates between:
#' \enumerate{
#'   \item E-step: computing the expectation of the complete log-likelihood function.
#'   \item M-step: maximizing the expectation of the complete log-likelihood function.
#' }
#' @examples
#' set.seed(1)
#' x <- c(rnorm(100, -2, 1), rnorm(100, 2, 1))
#' fit <- EM(x, K = 2)
#' fit$mu
#' fit$pi
#'
#' @export EM
#' @importFrom stats dnorm kmeans sd
EM <- function(X, K = 2, max_iter = 100, tol = 1e-5) {

  # Para evitar problemas por valores extremos
  eps_sd  <- sqrt(.Machine$double.eps)   # para evitar sd=0
  eps_den <- .Machine$double.xmin        # para evitar denom=0 y log(0)

  n <- length(X)
  num_iteraciones <- 0
  log_likelihood_old <- -Inf

  # Inicializo con K-means
  kmeans_result <- kmeans(X, centers = K, nstart = 20)
  mu_k <- as.numeric(kmeans_result$centers)
  pi_k <- as.numeric(table(kmeans_result$cluster) / n)
  sigma_k <- numeric(K)

  for (k in 1:K) {
    cluster_points <- X[kmeans_result$cluster == k]
    # Evitamos cluster de un punto y sd = 0
    if (length(cluster_points) <= 1L) {
      sigma_k[k] <- sd(X)
    } else {
      sigma_k[k] <- sd(cluster_points) *
        sqrt((length(cluster_points) - 1) / (length(cluster_points)))
    }
    if (!is.finite(sigma_k[k]) || sigma_k[k] <= 0) sigma_k[k] <- sd(X)
    if (!is.finite(sigma_k[k]) || sigma_k[k] <= 0) sigma_k[k] <- eps_sd
  }

  for (iter in 1:max_iter) {

    # E-step
    q <- matrix(0, n, K)

    for (i in 1:n) {
      dens <- pi_k * dnorm(X[i], mean = mu_k, sd = pmax(sigma_k, eps_sd))
      denom <- sum(dens)
      # Evita dividir por 0 / NaN
      if (!is.finite(denom) || denom <= eps_den) {
        q[i, ] <- rep(1 / K, K)
      } else {
        q[i, ] <- dens / denom
      }
    }

    # M-step
    N_k <- colSums(q)

    # Evita N_k=0 que da NaN en actualizaciones
    if (any(N_k <= eps_den)) {
      N_k <- pmax(N_k, eps_den)
      q <- sweep(q, 2, colSums(q), "/")
      q[!is.finite(q)] <- 1 / K
      N_k <- colSums(q)
    }

    # Actualizo
    for (k in 1:K) {
      mu_k[k] <- sum(q[, k] * X) / N_k[k]
      sigma_k[k] <- sqrt(sum(q[, k] * (X - mu_k[k])^2) / N_k[k])
      if (!is.finite(sigma_k[k]) || sigma_k[k] <= 0) sigma_k[k] <- eps_sd
    }

    pi_k <- N_k / n
    # clamp numérico de pi para evitar ceros exactos
    pi_k <- pmax(pi_k, eps_den)
    pi_k <- pi_k / sum(pi_k)

    # Nueva log-verosimilitud
    log_likelihood_new <- 0
    for (i in 1:n) {
      s <- sum(pi_k * dnorm(X[i], mean = mu_k, sd = pmax(sigma_k, eps_sd)))
      log_likelihood_new <- log_likelihood_new + log(pmax(s, eps_den))
    }

    # Puede haber NA
    if (is.na(log_likelihood_new) || is.infinite(log_likelihood_new)) {
      next
    }

    # Verificamos convergencia
    if (abs(log_likelihood_new - log_likelihood_old) < tol) {
      break
    }

    log_likelihood_old <- log_likelihood_new
    num_iteraciones <- num_iteraciones + 1
  }

  return(list(
    mu = mu_k,
    sigma = sigma_k,
    pi = pi_k,
    num_iteraciones = num_iteraciones,
    posterior = q
  ))
}
