# ModAtanExp.R
#' Modified Atan Exponential Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Modified Atan Exponential distribution.
#'
#' The Modified Atan Exponential distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Modified Atan Exponential distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \left[\frac{2}{\pi} \arctan \left\{\beta x e^{\alpha x}\right\}\right]
#' ^\lambda ; x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dmod.atan.exp()} — Density function
#'   \item \code{pmod.atan.exp()} — Distribution function
#'   \item \code{qmod.atan.exp()} — Quantile function
#'   \item \code{rmod.atan.exp()} — Random generation
#'   \item \code{hmod.atan.exp()} — Hazard function
#' }
#'
#' @usage
#' dmod.atan.exp(x, alpha, beta, lambda, log = FALSE)
#' pmod.atan.exp(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qmod.atan.exp(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rmod.atan.exp(n, alpha, beta, lambda)
#' hmod.atan.exp(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dmod.atan.exp}: numeric vector of (log-)densities
#'   \item \code{pmod.atan.exp}: numeric vector of probabilities
#'   \item \code{qmod.atan.exp}: numeric vector of quantiles
#'   \item \code{rmod.atan.exp}: numeric vector of random variates
#'   \item \code{hmod.atan.exp}: numeric vector of hazard values
#' }
#'
#' @references
#' Chaudhary, A.K., Telee, L.B.S., & Kumar, V.(2023). 
#' Modified Arctan Exponential Distribution with application to COVID-19 
#' Second Wave data in Nepal. \emph{Journal of Econometrics and Statistics}, 
#' \bold{4(1)}, 63--78.
#'
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dmod.atan.exp(x, 0.1, 0.2, 1.2)
#' pmod.atan.exp(x, 0.1, 0.2, 1.2)
#' qmod.atan.exp(0.5, 0.1, 0.2, 1.2)
#' rmod.atan.exp(10, 0.1, 0.2, 1.2)
#' hmod.atan.exp(x, 0.1, 0.2, 1.2)
#' 
#' # Data
#' x <- bladder
#' # ML estimates
#' params = list(alpha=0.04599, beta=0.14935, lambda=1.23266)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pmod.atan.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qmod.atan.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dmod.atan.exp, pfun=pmod.atan.exp, plot=FALSE)
#' print.gofic(out)
#' 
#' @name ModAtanExp
#' @export
NULL

#' @rdname ModAtanExp
#' @usage NULL
#' @export
dmod.atan.exp <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	u <- exp(alpha * x[valid])
	w <- beta * x[valid] * u 	
	z <- atan(w) ^ (lambda-1)
	t <- (2/pi) ^ lambda 
    pdf[valid] <- t * beta * lambda * u * (1+alpha*x[valid]) * z * (1/(1+w^2))
    }
	if (log) 
        pdf <- log(pdf)
    return(pdf)   
}

#' @rdname ModAtanExp
#' @usage NULL
#' @export
pmod.atan.exp <- function (q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
	z <- atan(beta * q[valid] * exp(alpha * q[valid]))		
    cdf[valid] <- ((2/pi) * z) ^ lambda
	}
    if (!lower.tail) 
        cdf <- 1.0 - cdf 
    if (log.p) 
        cdf <- log(cdf)    
    return(cdf)   
}

#' @rdname ModAtanExp
#' @usage NULL
#' @export
qmod.atan.exp <- function (p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	beta * x * exp(alpha * x)-tan(0.5 * pi * (wi ^ (1/lambda)))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname ModAtanExp
#' @usage NULL
#' @export
rmod.atan.exp <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qmod.atan.exp(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ModAtanExp
#' @usage NULL
#' @export
hmod.atan.exp <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
	nume <- dmod.atan.exp(x, alpha, beta, lambda) 
	surv <- 1.0 - pmod.atan.exp(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)   
} 

