# LogisInvLomax.R
#' Logistic Inverted Lomax Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic Inverted Lomax distribution.
#'
#' The Logistic Inverted Lomax distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic Inverted Lomax distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1 - {\left[ {1 + {{\left\{ {{{\left( {1 + \frac{\beta }{x}} \right)}
#' ^\lambda } - 1} \right\}}^\alpha }} \right]^{ - 1}}\quad ;\,x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.inv.lomax()} — Density function
#'   \item \code{plogis.inv.lomax()} — Distribution function
#'   \item \code{qlogis.inv.lomax()} — Quantile function
#'   \item \code{rlogis.inv.lomax()} — Random generation
#'   \item \code{hlogis.inv.lomax()} — Hazard function
#' }
#'
#' @usage
#' dlogis.inv.lomax(x, alpha, beta, lambda, log = FALSE)
#' plogis.inv.lomax(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.inv.lomax(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.inv.lomax(n, alpha, beta, lambda)
#' hlogis.inv.lomax(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.inv.lomax}: numeric vector of (log-)densities
#'   \item \code{plogis.inv.lomax}: numeric vector of probabilities
#'   \item \code{qlogis.inv.lomax}: numeric vector of quantiles
#'   \item \code{rlogis.inv.lomax}: numeric vector of random variates
#'   \item \code{hlogis.inv.lomax}: numeric vector of hazard values
#' }
#'
#' @references
#' Joshi, R. K., & Kumar, V. (2021). 
#' The Logistic Inverse Lomax Distribution with Properties and Applications. 
#' \emph{International Journal of Mathematics and Computer Research}, 
#' \bold{9(1)}, 2169--2177. \doi{10.47191/ijmcr/v9i1.02}  
#'
#' Lan, Y., & Leemis, L. M. (2008). The Logistic-Exponential Survival Distribution. 
#' \emph{Naval Research Logistics},  \bold{55},  252--264.
#'  
#' @examples
#' x <- seq(0.1, 10, 0.2)
#' dlogis.inv.lomax(x, 2.0, 5.0, 0.2)
#' plogis.inv.lomax(x, 2.0, 5.0, 0.2)
#' qlogis.inv.lomax(0.5, 2.0, 5.0, 0.2)
#' rlogis.inv.lomax(10, 2.0, 5.0, 0.2)
#' hlogis.inv.lomax(x, 2.0, 5.0, 0.2)
#' 
#' # Data
#' x <- bladder
#' # ML estimates
#' params = list(alpha=2.87951, beta=38.51405, lambda=0.35313)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.inv.lomax, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.inv.lomax, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.inv.lomax, pfun=plogis.inv.lomax, plot=FALSE)
#' print.gofic(out)
#' 
#' @name LogisInvLomax
#' @export
NULL

#' @rdname LogisInvLomax
#' @usage NULL
#' @export  
dlogis.inv.lomax <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {	
    u <- (1+(beta/x[valid]))
	w <- u ^ lambda	
    num <- alpha * beta * lambda*(x[valid]^-2) *(u^(lambda-1)) *((w-1)^(alpha-1))
    deno <- (1.0 + (w - 1.0) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname LogisInvLomax
#' @usage NULL
#' @export  
plogis.inv.lomax <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{ 
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- (1 + (beta/q[valid])) ^ lambda 
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0/tmp
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisInvLomax
#' @usage NULL
#' @export 
qlogis.inv.lomax <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
	qtl[valid] <- beta / ((((((1-p[valid])/p[valid]) ^ (1/alpha))+1) ^(1/lambda))-1) 	   
	}
    return(qtl)   
 } 

#' @rdname LogisInvLomax
#' @usage NULL
#' @export 
rlogis.inv.lomax <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.inv.lomax(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisInvLomax
#' @usage NULL
#' @export 
hlogis.inv.lomax <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dlogis.inv.lomax(x, alpha, beta, lambda, log=FALSE)
	surv <- 1.0 - plogis.inv.lomax(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}

