# InvSGZ.R
#' Inverted Shifted Gompertz (ISG) Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverted Shifted Gompertz distribution.
#'
#' The Inverted Shifted Gompertz distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, and \eqn{\theta > 0}.
#'
#' @details
#' The Inverted Shifted Gompertz distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \theta) =
#' 1-\left(1-e^{-\theta / x}\right) \exp \left(-\alpha e^{-\theta / x}\right)  \quad ;\;x > 0.
#' }
#'
#' where\eqn{\alpha} and \eqn{\theta} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dinv.sgz()} — Density function
#'   \item \code{pinv.sgz()} — Distribution function
#'   \item \code{qinv.sgz()} — Quantile function
#'   \item \code{rinv.sgz()} — Random generation
#'   \item \code{hinv.sgz()} — Hazard function
#' }
#'
#' @usage
#' dinv.sgz(x, alpha, theta, log = FALSE)
#' pinv.sgz(q, alpha, theta, lower.tail = TRUE, log.p = FALSE)
#' qinv.sgz(p, alpha, theta, lower.tail = TRUE, log.p = FALSE)
#' rinv.sgz(n, alpha, theta)
#' hinv.sgz(x, alpha, theta)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param theta positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.sgz}: numeric vector of (log-)densities
#'   \item \code{pinv.sgz}: numeric vector of probabilities
#'   \item \code{qinv.sgz}: numeric vector of quantiles
#'   \item \code{rinv.sgz}: numeric vector of random variates
#'   \item \code{hinv.sgz}: numeric vector of hazard values
#' }
#'
#' @references   
#' Chaudhary, A.K., Sapkota, L.P., & Kumar, V.  (2020). 
#' Inverted Shifted Gompertz Distribution with Theory and Applications. 
#' \emph{Pravaha}, \bold{26(1)}, 1--10. \doi{10.3126/pravaha.v26i1.41645}
#'    
#' Jimenez T.F. (2014). Estimation of the Parameters of the Shifted Gompertz 
#' Distribution, Using Least Squares, Maximum Likelihood and Moments Methods. 
#' \emph{Journal of Computational and Applied Mathematics}, \bold{255(1)} 867--877.
#'
#' @examples
#' x <- seq(1.0, 5, 0.25)
#' dinv.sgz(x, 25, 10)
#' pinv.sgz(x, 25, 10)
#' qinv.sgz(0.5, 25, 10)
#' rinv.sgz(10, 25, 10)
#' hinv.sgz(x, 25, 10)
#' 
#' # Data
#' x <- fibers65
#' # ML estimates
#' params = list(alpha=215.8181, theta=12.7678)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.sgz, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.sgz, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dinv.sgz, pfun=pinv.sgz, plot=FALSE)
#' print.gofic(out)
#' 
#' @name InvSGZ
#' @aliases dinv.sgz pinv.sgz qinv.sgz rinv.sgz hinv.sgz
#' @export
NULL

#' @rdname InvSGZ
#' @usage NULL
#' @export   
dinv.sgz <- function (x, alpha, theta, log = FALSE)
{
	if (!all(sapply(list(x, alpha, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, theta) <= 0)) {
		stop("Parameters 'alpha' and 'theta' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
    u <- theta/ x[valid]
    pdf[valid] <- theta * (x[valid]^-2) * exp(-u) * exp(-alpha * exp(-u)) * (1 + alpha * (1 - exp(-u)))   
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname InvSGZ
#' @usage NULL
#' @export
pinv.sgz <- function (q, alpha, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, theta), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, theta) <= 0)) {
		stop("Parameters 'alpha' and 'theta' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(-theta / q[valid])
    cdf[valid] <- 1.0 - (1.0 - u) * exp( - alpha * u)   
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname InvSGZ
#' @usage NULL
#' @export
qinv.sgz <- function (p, alpha, theta, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, theta) <= 0))
		stop("Parameters 'alpha', and 'theta' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) log(1 - exp(-theta / x)) - alpha * exp(-theta / x) - log(1-wi)   

	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   # small positive
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
}

#' @rdname InvSGZ
#' @usage NULL
#' @export
rinv.sgz <- function(n, alpha, theta)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, theta), is.numeric)))
		stop("'alpha', and 'theta' must be numeric.")
	if (any(c(alpha, theta) <= 0))
		stop("Invalid parameters: alpha, and theta must be positive.")

    u <- runif(n)
    q <- qinv.sgz(u, alpha, theta)
    return(q)
}

#' @rdname InvSGZ
#' @usage NULL
#' @export
hinv.sgz <- function (x, alpha, theta)
{
    if (!all(sapply(list(x, alpha, theta), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, theta, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dinv.sgz(x, alpha, theta) 
    surv <- 1.0 - pinv.sgz(x, alpha, theta)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 



