% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/norma_gene.R
\name{norma_gene}
\alias{norma_gene}
\title{Normalize CT values using NORMAgene}
\usage{
norma_gene(data, conditions = NULL, replicates = NULL, ct_values = NULL)
}
\arguments{
\item{data}{A data frame structured with biological replicates in rows, and experimental metadata and gene-wise raw CT values in columns.}

\item{conditions}{A single column name in \code{data} specifying experimental condition membership in the case of a multi-condition experiment, or \code{NULL} in the case of a single condition experiment. Normalization is applied within experimental conditions when specified, or across all replicates when \code{NULL}. This argument must be explicitly provided.}

\item{replicates}{A single column name in \code{data} containing replicate identifiers, or \code{NULL} if replicate identifiers are not present. If provided, replicate identifiers are used for naming of outputs only, and are not used in normalization calculations. This argument must be explicitly provided.}

\item{ct_values}{Optional character vector specifying column names in \code{data} containing CT values to be normalized. If \code{NULL}, all numeric columns except \code{conditions} and \code{replicates} are used.}
}
\value{
A data frame with the same organization as \code{data} containing normalized CT values and any provided experimental metadata. The per-replicate scaling factors used for normalization, as well as within gene and within experimental condition relative variability metrics, are attached as attributes.
}
\description{
Applies the least squares fit-based NORMAgene data-driven normalization algorithm originally described by Heckmann et al. (2011) to raw CT values provided via an input data frame appended with experimental meta-data, and returns a data frame containing normalized CT values with scaling factors and relative variability metrics attached as attributes.
}
\details{
Users must explicitly specify how experimental conditions and replicate identifiers are handled to avoid accidental normalization of numeric metadata. Scaling factors can be retrieved from the output object using \code{\link[=correction_factors]{correction_factors()}}. Relative variability metrics can be retrieved from the output object using \code{\link[=relative_variability]{relative_variability()}}. For more information on the NORMAgene algorithm or relative variability metrics, see \link{NORMAgene-package}.
}
\examples{
# USE-CASE WITH MULTIPLE EXPERIMENTAL CONDITIONS

# load example dataset containing raw CT values and
# metadata from a multi-condition experiment

data(multi_cond_data)
raw_data<-multi_cond_data

#normalize CT values via NORMAgene

norm_data<-norma_gene(
  data = raw_data,
  conditions = "Diagnosis",
  replicates= "Sample_id"
)

# retrieve relative variability metrics

relative_variability(norm_data, type = "by_gene")
relative_variability(norm_data, type = "by_condition")

# USE-CASE WITH a SINGLE EXPERIMENTAL CONDITION

# load example dataset containing raw CT values and
# metadata from a single-condition experiment

data(single_cond_data)
raw_data<-single_cond_data

#normalize CT values via NORMAgene

norm_data<-norma_gene(
  data = raw_data,
  conditions = NULL,
  replicates= "Sample_id"
)

# retrieve relative variability metrics

relative_variability(norm_data, type = "by_gene")
relative_variability(norm_data, type = "by_condition")

}
\references{
Heckmann, LH., Sørensen, PB., Krogh, PH., & Sørensen, JG. (2011).
NORMA-Gene: a simple and robust method for qPCR normalization based on target gene data.
\emph{BMC Bioinformatics}, 12, 250.
\doi{10.1186/1471-2105-12-250}
}
\seealso{
\code{\link[=correction_factors]{correction_factors()}} \cr
\code{\link[=relative_variability]{relative_variability()}} \cr
\link{NORMAgene-package}
}
