#' Retrieve scaling factors from NORMAgene output
#'
#' Retrieves the per-replicate scaling factors used for normalization.
#'
#' @param x
#' An object returned by [norma_gene()].
#'
#' @return
#' A numeric vector of correction factors. If replicate identifiers were passed to [norma_gene()], the vector is named accordingly.
#'
#' @seealso
#' [norma_gene()]
#'
#' @examples
#' # load example dataset containing raw CT values and
#' # metadata from a multi-condition experiment
#'
#' data(multi_cond_data)
#' raw_data<-multi_cond_data
#'
#' #normalize CT values via NORMAgene
#'
#' norm_data<-norma_gene(
#'   data = raw_data,
#'   conditions = "Diagnosis",
#'   replicates= "Sample_id"
#' )
#'
#' # retrieve scaling factors
#'
#' correction_factors(norm_data)
#'
#' @export

correction_factors<-function(x){

  #----input parsing and validation----

  if(!inherits(x, "norma_gene")){
    stop("`x` must be an object returned by `norma_gene()`.")
  }

  cor_fact<-attr(x, "cor_fact", exact=TRUE)

  if(is.null(cor_fact)){
    stop("No correction factors found in `x`.")
  }

  #----return----

  cor_fact
}
