#===============================================================================
#                      MOT Test Functions
#===============================================================================

#' @title Multivariate Overall Trend Test
#'
#' @description Computes the test statistic for the multivariate overall trend
#'  (MOT) test and the p-value.
#'
#' @param data Numeric matrix, a two dimensional data set where each column is a
#'  variable of the series.
#' @param covar Numeric vector, time order of the data series. If none given \eqn{covar = 1, 2, ..., n}.
#' @param width Integer, the width to be used for the *moving window* algorithm.
#'  Should be bigger than 1 and smaller than the number of observations, see **details**.
#' @param alpha Numeric value, significance level for the test. Must be between 0 and 1, default is 0.05
#' @param Nbs Positive Integer, number of samples to be used for bootstrapping. Default is 1000.
#'
#' @return A named list:
#' - `statistic`, computed MOT test statistic.
#' - `p.value`, computed *p-value* computed using bootstrap.
#' - `result`, the result of the test. If `TRUE` the trend is considered significant.
#'
#' @details
#' The multivariate overall trend (MOT) test allows to check for trend in
#' the marginals and dependence structure of a multivariate distribution.
#' The moving window technique has been employed in order to take into account the dependence evolution,
#' represented by Kendall's \eqn{\tau}, according to time.
#'
#' The test evaluates a null hypothesis of no trend in the data series,
#' against an alternative hypothesis of a monotonic trend.
#' Goutali and Chebana (2024) propose the following test statistic:
#'
#' Let \eqn{X = (x_1, x_2, ..., x_n)} and \eqn{Y = (y_1, y_2, ..., y_n)} be variables in a data series,
#' and \eqn{T} the time order, the test statistic is given by:
#'
#' \deqn{T_{MOT}=\frac{1}{3}\left(\tau_n(X,T)^2 + \tau_n(Y,T)^2 + \tau_n(\tau_{nw}(X,Y),T')^2  \right)}
#'
#' where \eqn{\tau_n} is the empirical version of bivariate Kendall's \eqn{\tau}. The
#' series of the Kendall's \eqn{\tau} obtained through moving window is \eqn{\tau_{nw}(X, Y)},
#' and \eqn{T'} is the time order of the moving window series.
#'
#' The choice of `width` is a trade-off. A small \eqn{w} increases the number of
#' rolling windows for reliable analysis, while a large \eqn{w} is necessary to have
#' sufficient values to identify the dependence structure. The p-value is computed
#' using a bootstrap procedure.
#'
#' @seealso
#' \itemize{
#'  \item `kendall.tau`: Function from the package `VGAM` used for computing the bivariate Kendall's \eqn{\tau}.
#'  \item `rollapply`: Function from the package `zoo` used to apply the moving window technique.
#'  \item `samp.bootstrap`: Function from the `resample` package, used to generate the samples
#'  necessary to perform bootstrapping.
#' }
#'
#' @references
#' Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
#' Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
#'
#' @examples
#' \donttest{
#' # CASE 1: Only trend in the dependence structure
#' # Sample data:
#' DependenceStructure <- generate_data("dependenceStructure", n = 50)
#'
#' width <- 10
#'
#' # Perform the mot test:
#' mot <- MOT_test(DependenceStructure, covar = NULL, width, alpha = 0.05, Nbs = 1000)
#' print(mot)
#'
#'
#' # CASE 2: Only trend in the marginal distributions
#' # Sample data:
#' MarginalTrend <- generate_data("marginalTrend", n = 50)
#'
#' # Perform the mot test:
#' mot <- MOT_test(MarginalTrend, width = width)
#' print(mot)
#'
#'
#' # CASE 3: No trend
#' # Sample data:
#' NoTrend <- generate_data("noTrend", n = 50)
#'
#' # Perform the mot test:
#' mot <- MOT_test(NoTrend, width = width)
#' print(mot)
#' }
#'
#'
#' @importFrom zoo rollapply
#' @importFrom VGAM kendall.tau
#' @importFrom resample samp.bootstrap
#'
#' @export

MOT_test <- function(data,
                     covar = NULL,
                     width,
                     alpha = 0.05,
                     Nbs = 1000) {}

MOT_test <- function(data,
                     covar = NULL,
                     width,
                     alpha = 0.05,
                     Nbs = 1000) {

  data_ <- as.matrix(data)

  # Input checks:

  if (!is.numeric(data_)) {
    stop("'data' must be numeric")
  }

  if (!is.numeric(width)) {
    stop("Parameter 'width' must be numeric")
  }

  if (!is.numeric(alpha)) {
    stop("Parameter 'alpha' must be numeric")
  }

  if (!is.numeric(Nbs)) {
    stop("Parameter 'Nbs' must be numeric")
  }

  # Series length
  n = nrow(data_)

  if (is.null(covar)) {
    covar = seq(1:n)
  }
  if (length(covar) != n) {
    stop("The length of data and covar must be the same")
  }

  if (width < 1 || width > n) {
    stop("Parameter 'width' must be bigger than 1 and smaller than number of observations")
  } else if (width == 1 || width == n) {
    warning("Inadequate value for 'width' parameter, must be bigger than 1 and smaller than number of observations")
  }

  if ((Nbs %% 1) != 0 || Nbs <= 0) {
    stop("Parameter 'Nbs' must be a positive integer.")
  }

  if (alpha <= 0 || alpha >= 1) {
    stop("Parameter 'alpha' must be between 0 and 1")
  }

  results <- MOT_pvalue(data_, covar, width, alpha, Nbs)
  return(results)
}

#===============================================================================
# MOT STATISTIC
MOT_statistic <- function(data, covar, width) {

  # Variable declaration
  T2 = 1:(nrow(data) - width + 1)

  # Defining 'kendall.tau' as a function
  kend <- function(x) {kendall.tau(x[, 1], x[, 2])}

  # Getting correlation coefficients between 'x' and 'y' in different subsets,
  # moving or rolling window

  roll = rollapply(data, width = width, FUN = kend,
                   by.column = FALSE, align = "right")

  X <- as.numeric(data[, 1])
  Y <- as.numeric(data[, 2])

  # Getting the three Kendall's taus
  dep1 = kendall.tau(as.numeric(roll), T2)^2
  dep2 = kendall.tau(X, covar)^2
  dep3 = kendall.tau(Y, covar)^2

  # Computing the test statistic
  tstat = (sum(dep1, dep2, dep3))/3

  return(tstat)
}

#===============================================================================
#MOT PVALUE
MOT_pvalue <- function(data, covar, width, alpha, Nbs) {

  ## Number of observations
  n <- nrow(data)

  # Test statistic storage
  Sn = rep(0, Nbs)

  # Matrix to be used as indices for the resampling with replacement
  index = samp.bootstrap(n, Nbs)

  # p-value estimation via bootstrap
  for (i in 1:Nbs) {
    dat_sample = data[index[, i],]
    Sn[i] = MOT_statistic(dat_sample, covar, width)
  }

  thresh <- MOT_statistic(data, covar, width)

  pvalue = sum(Sn > rep(thresh, Nbs)) / Nbs

  if (pvalue < alpha) {
    result = TRUE
  } else {
    result = FALSE
  }

  return(list("statistic" = thresh ,"p.value" = pvalue, "result" = result))
}

