#' Fit the GM (1, 1) model for multivariate time series data
#'
#' @description The multigreyfit function fits the GM (1,1) model for multivariate time series data and computes its model parameters as well as different model evaluation criteria.
#'
#' @param data Input multivariate time series data.
#'
#' @return A list with the following components:
#' \item{a}{Control variable, i.e., parameter of fitted GM (1, 1) model for each of the time series variables.}
#' \item{b}{Developing coefficient, i.e., parameter of fitted GM (1, 1) model for each of the time series variables.}
#' \item{mse}{Mean Squared Error (MSE) for each time series variable.}
#' \item{rmse}{Root Mean Square Error (RMSE) of fitted GM (1, 1) model for each of the time series variables}
#' \item{mape}{Relative Mean Absolute Percentage Error (MAPE) of fitted GM (1, 1) model for each of the time series variables}
#' \item{mae}{Mean Absolute Error (MAE) of fitted GM (1, 1) model for each of the time series variables}
#'
#' @details This function returns the parameters of the fitted grey model, i.e., a and b which are known as control variable
#'  and developing coefficient respectively for each of the of the variables in the multivariate timeseries data. This function
#'  also returns the mean square error (MSE), mean absolute error (MAE), root mean square error (RMSE) and relative mean absolute percentage error (MAPE)
#'   values of the fitted GM (1, 1) model for each of the variables in the multivariate time series data.
#'
#' @references
#' Akay, D. and Atak, M. (2007). Grey prediction with rolling mechanism for electricity demand forecasting of Turkey.
#' \emph{Energy}, 32(9), 1670-1675.<DOI:10.1016/j.energy.2006.11.014>
#'
#' Deng, J. (1989). Introduction to grey system theory. \emph{The Journal of Grey System}, 1(1), 1-24.
#'
#' Hsu, L.C. and Wang, C.H. (2007). Forecasting the output of integrated circuit industry using a grey model improved by
#' Bayesian analysis. \emph{Technological Forecasting and Social Change}, 74(6), 843-853.<DOI:10.1016/j.techfore.2006.02.005>
#'
#' @examples
#' # Example data
#' xt <- c(640, 684, 713, 745, 809, 811, 883, 893, 904, 935, 1044, 1069)
#' yt <- c(50, 64, 93, 113, 131, 152, 164, 201, 224, 268, 286, 290)
#' zt  <- c(550,504,493,413,431,352,364,301,294,268,286,230)
#' data <- cbind(xt, yt, zt)
#'
#' # Apply the multigreyfit function
#' multigreyfit(data)
#' @export
#' @importFrom zoo rollapply
#' @importFrom stats lm

multigreyfit=function(data)
{
  datat=t(data)
  d=dim(datat)[1]
  n=dim(datat)[2]
  #library(zoo)
  xdatat=matrix(0,d,n)
  z=matrix(0,d,n-1)
  x=matrix(0,d,n-1)
  p=matrix(0,d,2)

  xpred=matrix(0,d,n-1)
  a=rep(0,d)
  b=rep(0,d)
  mse=rep(0,d)
  mae=rep(0,d)
  rmse=rep(0,d)
  mape=rep(0,d)

  for (s in 1:d) {
    xdatat[s,]=as.numeric(datat[s,])
    z[s,]=zoo::rollapply(cumsum(xdatat[s,]),2,mean)
    x[s,]=xdatat[s,-1]
    p[s,]=(lm(x[s,]~z[s,]))$coefficients

    for (i in 1:n-1) {

      xpred[s,i]=(xdatat[s,1]-(p[s,1]/(-p[s,2])))*exp(p[s,2]*i)*(1-exp(-p[s,2]))
    }
    a[s]=p[s,1]
    b[s]=p[s,2]
    mse[s]=mean((x[s,]-xpred[s,])^2)
    mae[s]=mean(abs(x[s,]-xpred[s,]))
    rmse[s]=sqrt(mean((x[s,]-xpred[s,])^2))
    mape[s]=mean((abs(x[s,]-xpred[s,])/x[s,]))*100
  }

  output <- list("a"=a, "b"=b, "mse"=mse,"mae"=mae,"rmse"=rmse,"mape"=mape)
  return(output)
}
