% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcregions.R
\name{calcregions}
\alias{calcregions}
\alias{addregions}
\alias{summary.regions_results}
\alias{ncombos}
\title{Fit segmented regression models for all combinations of breakpoints}
\usage{
calcregions(
  pco,
  scores,
  noregions,
  minvert = 3,
  cont = TRUE,
  exhaus = TRUE,
  includebp = NULL,
  omitbp = NULL,
  ncombos_file_trigger = 1e+07,
  temp_file_dir = tempdir(TRUE),
  cl = NULL,
  verbose = TRUE
)

addregions(
  regions_results,
  noregions,
  exhaus = TRUE,
  ncombos_file_trigger = 1e+07,
  temp_file_dir = tempdir(TRUE),
  cl = NULL,
  verbose = TRUE
)

\method{summary}{regions_results}(object, ...)

ncombos(pco, noregions, minvert = 3, includebp = NULL, omitbp = NULL)
}
\arguments{
\item{pco}{a \code{regions_pco} object; the output of a call to \code{\link[=svdPCO]{svdPCO()}}.}

\item{scores}{\code{numeric}; the indices of the PCO scores to use as the outcomes in fitting the models (e.g., \code{1:4} to use the first four scores). Can also be the output of a call to \code{\link[=PCOselect]{PCOselect()}}.}

\item{noregions}{\code{numeric}; for \code{calcregions()}, the maximum number of regions for which models are fit (e.g, 4 to request models with 1 to 4 regions); for \code{addregions()}, a vector containing the numbers of regions to add (e.g., 5:6 to request models with 5 and 6 regions); for \code{ncombos()}, a vector containing the numbers of regions to check.}

\item{minvert}{\code{numeric}; the minimum number of vertebrae allowed in each region. Default is 3.}

\item{cont}{\code{logical}; whether to fit models that are continuous (\code{TRUE}) or discontinuous (\code{FALSE}) at the breakpoints. Default is \code{TRUE}.}

\item{exhaus}{\code{logical}; whether to fit all possible models (\code{TRUE}) or use heuristics to reduce the number of models fit (\code{FALSE}). Default is \code{TRUE}. See Details. Setting to \code{FALSE} can reduce the size of the resulting object.}

\item{includebp}{an optional vector of vertebrae that must be included in any tested set of breakpoints, e.g., if it is known that two regions are divided at that vertebra. \code{includebp} does not have to obey the \code{minvert} rules, but a warning will be thrown if it doesn't.}

\item{omitbp}{an optional vector of vertebrae to be omitted from the list of possible breakpoints, e.g., if it is known that two adjacent vertebrae belong to the same region.}

\item{ncombos_file_trigger}{\code{numeric}; when the number of eligible combinations of breakpoints exceeds this number, the problem will be split into smaller problems, with the results of each stored in its own temporary file in the directory supplied to \code{temp_file_dir} before being re-read into memory. The primary purpose of this is to preserve memory when \code{exhaus = FALSE} by delegating storage of the results to disk instead of RAM.}

\item{temp_file_dir}{string; the directory where the temporary files will be saved (and then deleted) when the number of breakpoint combinations exceeds \code{ncombos_file_trigger}. Default is the directory produced by \code{\link[=tempdir]{tempdir()}}, but it is much safer to provide your own directory, which must already exist on your machine. See Details.}

\item{cl}{a cluster object created by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}, an integer to indicate number of child-processes (integer values are ignored on Windows) for parallel evaluations, or \code{"future"} to use a future backend. \code{NULL} (the default) refers to sequential evaluation (no parallelization). See \code{\link[pbapply:pbapply]{pbapply::pbapply()}} for details.}

\item{verbose}{\code{logical}; whether to print information about the fitting process, including a progress bar. Default is \code{TRUE}.}

\item{regions_results, object}{a \code{regions_results} object; the output of a call to \code{calcregions()} or \code{addregions()}.}

\item{\dots}{ignored.}
}
\value{
A \code{regions_results} object with the following components:
\itemize{
\item \code{results} - the results of the fitting process for each combination of breakpoints
\item \code{stats} - statistics summarizing the fitting process. Use \code{summary()} to view this information in a clean format.
}

\code{ncombos()} returns a numeric vector with the number of breakpoint combinations for each number of regions (which are stored as the names).
}
\description{
\code{calcregions()} enumerates all possible combinations of breakpoints to fit multivariate segmented regression models. \code{addregions()} adds models with additional numbers of regions to the resulting output object. \code{ncombos()} computes an upper bound on the number of breakpoint combinations that will be tested.
}
\details{
\code{calcregions()} enumerates all possible combinations of breakpoints that satisfy the constraint imposed by \code{minvert} (i.e., that breakpoints need to be at least \code{minvert} vertebrae apart) and fits the segmented regression models implied by each combination. These are multivariate regression models with the PCO scores specified by \code{scores} as the outcomes. When \code{cont = TRUE}, these regression models are continuous; i.e., the regression lines for each region connect at the breakpoints. Otherwise, the models are discontinuous so that each region has its own intercept and slope. The models are fit using \code{\link[=.lm.fit]{.lm.fit()}}, which efficiently implements ordinary least squares regression.

When \code{exhaus = FALSE}, heuristics are used to reduce the number of models to fit, which can be useful for keeping the size of the resulting object down by avoiding fitting models corresponding to breakpoint combinations that yield a poor fit to the data. Only breakpoint combinations that correspond to the breakpoints of the best fitting model with a smaller number of regions +/- 3 vertebrae are used, and only models that have an RSS smaller than half a standard deviation more the smallest RSS are kept.

\code{addregions()} should be used on an existing \code{regions_results} object to add models with more regions. Internally, it works just the same as \code{calcregions()}.

\code{ncomobs()} computes an upper bound on the number of possible breakpoint combinations. When \code{exhaus = FALSE} or \code{includebp} is specified, the actual number of combinations will be smaller than that produced by \code{ncombos()}.

When the number of possible combinations of breakpoints for a given number of regions (as computed by \code{ncombos()}) is larger than \code{ncombos_file_trigger}, the problem will be split into smaller problems, with the results of each stored in temporary files that are deleted when the function completes. These temporary files will be stored in the directory supplied to \code{temp_file_dir}. By default, this is the temporary directory produced by \code{\link[=tempdir]{tempdir()}}. However, this directory can be deleted by R at any time without warning, which will cause the function to crash, so it is a good idea to supply your own directory that will be preserved. You can use \code{ncombos()} to check to see if the number of breakpoint combinations exceeds \code{ncombos_file_trigger}.
}
\examples{
data("alligator")

alligator_data <- process_measurements(alligator,
                                       pos = "Vertebra")

# Compute PCOs
alligator_PCO <- svdPCO(alligator_data)

# Fit segmented regression models for 1 to 5 regions
# using PCOs 1 to 4 and a continuous model with a
# non-exhaustive search
regionresults <- calcregions(alligator_PCO,
                             scores = 1:4,
                             noregions = 5,
                             minvert = 3,
                             cont = TRUE,
                             exhaus = FALSE,
                             verbose = FALSE)

regionresults

# View model fitting summary
summary(regionresults)

# Add additional regions to existing results,
# exhaustive search this time
regionresults <- addregions(regionresults,
                            noregions = 6:7,
                            exhaus = TRUE,
                            verbose = FALSE)

regionresults

summary(regionresults)

# Fit segmented regression models for 1 to 5 regions
# using PCOs 1 to 4 and a discontinuous model with a
# exhaustive search, excluding breakpoints at vertebrae
# 10 and 15
regionresults <- calcregions(alligator_PCO,
                             scores = 1:4,
                             noregions = 5,
                             minvert = 3,
                             cont = FALSE,
                             omitbp = c(10, 15),
                             verbose = FALSE)

regionresults

summary(regionresults)

# Compute the number of breakpoint combinations for given
# specification using `ncombos()`; if any number exceeds
# the value supplied to `ncombos_file_trigger`, results
# will temporary be stored in files before being read in and
# deleted.
ncombos(alligator_PCO,
        noregions = 1:8,
        minvert = 3)
}
\seealso{
\code{\link[=calcmodel]{calcmodel()}} to fit a segmented regression model for a single set of breakpoints; \code{\link[=modelselect]{modelselect()}} to select the best model for each number of regions based on RSS; \code{\link[=modelsupport]{modelsupport()}} to compute statistics the describe the support of the best models; \code{\link[=calcBPvar]{calcBPvar()}} to compute the variability in the optimal breakpoints.
}
