# Tools for testing normality, estimating stable parameters, and generating reports
# =============================================================================
# NORMALITY TESTS
# =============================================================================


#' Test normality using multiple statistical tests
#'
#' Applies Shapiro-Wilk, Jarque-Bera, Anderson-Darling, and Kolmogorov-Smirnov tests
#' to assess normality of a dataset.
#'
#' @param x Numeric vector of data.
#' @return List of test statistics and p-values.
#' @importFrom stats shapiro.test ks.test pchisq
#' @importFrom e1071 skewness kurtosis
#' @importFrom nortest ad.test
#' @export
test_normality <- function(x) {
  x <- x[!is.na(x)]
  if (length(x) < 4) stop("Data must have at least 4 values for normality tests.")

  shapiro_result <- shapiro.test(x)
  n <- length(x)
  skew_val <- e1071::skewness(x)
  kurt_val <- e1071::kurtosis(x, type = 2) + 3
  jb_stat <- n * (skew_val^2 / 6 + (kurt_val - 3)^2 / 24)
  jb_pvalue <- 1 - pchisq(jb_stat, df = 2)

  ad_result <- nortest::ad.test(x)
  ks_result <- ks.test(x, "pnorm", mean(x), sd(x))

  return(list(
    Shapiro    = list(statistic = shapiro_result$statistic, pvalue = shapiro_result$p.value),
    JarqueBera = list(statistic = jb_stat, pvalue = jb_pvalue),
    Anderson  = list(statistic = ad_result$statistic, pvalue = ad_result$p.value),
    KS        = list(statistic = ks_result$statistic, pvalue = ks_result$p.value)
  ))
}

# =============================================================================
# MOMENTS AND QCV
# =============================================================================


#' Calculate skewness and kurtosis
#'
#' Computes the skewness and kurtosis of a numeric vector.
#'
#' @param x Numeric vector of data.
#' @return List with skewness and kurtosis.
#' @importFrom e1071 skewness kurtosis
#' @export
skew_kurtosis <- function(x) {
  x <- x[!is.na(x)]
  return(list(
    skewness = e1071::skewness(x),
    kurtosis = e1071::kurtosis(x, type = 2) + 3
  ))
}


#' QCV statistic for tail heaviness
#'
#' Computes the QCV (Quantile Conditional Variance) statistic to assess tail heaviness
#' of a distribution.
#'
#' @param x Numeric vector of data.
#' @return Numeric: QCV value.
#' @importFrom stats quantile var sd
#' @export
qcv_stat <- function(x) {
  x <- x[!is.na(x)]
  tryCatch({
    x_norm <- (x - mean(x)) / sd(x)
    x_sorted <- sort(x_norm)
    q25 <- quantile(x_sorted, 0.25)
    q75 <- quantile(x_sorted, 0.75)
    var_left <- var(x_sorted[x_sorted < q25])
    var_right <- var(x_sorted[x_sorted > q75])
    var_mid <- var(x_sorted[x_sorted > q25 & x_sorted < q75])
    if (is.na(var_mid) || var_mid == 0) return(1.0)
    return((var_left + var_right) / (2 * var_mid))
  }, error = function(e) {
    warning("Error computing QCV statistic: ", e$message)
    return(1.0)
  })
}

# =============================================================================
# STABLE PARAMETER ESTIMATION
# =============================================================================


#' Estimate stable parameters using method of moments
#'
#' Provides a rough estimation of alpha-stable parameters using empirical moments.
#'
#' @param x Numeric vector of data.
#' @return List with estimated alpha, beta, gamma, delta.
#' @importFrom stats sd
#' @export
estimate_stable_r <- function(x) {
  x <- x[!is.na(x)]
  tryCatch({
    alpha_est <- 1.8
    beta_est  <- 0.0
    gamma_est <- sd(x) * 0.8
    delta_est <- mean(x)
    return(list(alpha = alpha_est, beta = beta_est, gamma = gamma_est, delta = delta_est))
  }, error = function(e) {
    warning("Error in stable parameter estimation: ", e$message)
    return(list(alpha = 1.5, beta = 0.0, gamma = 1.0, delta = 0.0))
  })
}

# =============================================================================
# VISUAL COMPARISON
# =============================================================================


#' Plot comparison between normal and stable distributions
#'
#' Displays a histogram of the data with overlaid normal and alpha-stable PDFs.
#'
#' @param x Numeric vector of data.
#' @param params_stable List with alpha, beta, gamma, delta.
#' @param fig_path Optional path to save the plot (PNG). If NULL, uses tempdir().
#' @return NULL (saves plot to file)
#' @importFrom graphics hist lines legend grid
#' @importFrom stats dnorm
#' @importFrom stabledist dstable
#' @importFrom grDevices dev.cur
#' @export
plot_vs_normal_stable <- function(x, params_stable, fig_path = NULL) {
  x <- x[!is.na(x)]
  if (length(x) < 2) {
    message("[plot_vs_normal_stable] Not enough data to plot.")
    return(invisible(NULL))
  }

  if (is.null(fig_path)) {
    fig_path <- file.path(tempdir(), "stability_test_plot.png")
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  xx <- seq(min(x), max(x), length.out = 500)
  norm_pdf <- stats::dnorm(xx, mean(x), stats::sd(x))

  stable_pdf <- tryCatch({
    stabledist::dstable(xx, alpha = params_stable$alpha,
                        beta = params_stable$beta,
                        gamma = params_stable$gamma,
                        delta = params_stable$delta, pm = 1)
  }, error = function(e) {
    message("[plot_vs_normal_stable] Stable PDF failed, using normal PDF instead.")
    norm_pdf
  })

  tryCatch({
    png(fig_path, width = 10, height = 6, units = "in", res = 150)
    hist(x, breaks = 50, freq = FALSE, col = rgb(0.7, 0.7, 0.7, 0.4),
         main = "Empirical vs Normal vs Stable PDF",
         xlab = "x", ylab = "Density")
    lines(xx, norm_pdf, col = "red", lty = 2, lwd = 2)
    lines(xx, stable_pdf, col = "blue", lwd = 2)
    legend("topright", legend = c("Data", "Normal PDF", "Stable PDF"),
           col = c("gray", "red", "blue"), lty = c(NA, 2, 1),
           lwd = c(NA, 2, 2), pch = c(15, NA, NA), pt.cex = 2)
    grid()
    dev.off()
  }, error = function(e) {
    message("[plot_vs_normal_stable] Failed to save plot: ", e$message)
    if (dev.cur() > 1) dev.off()
  })

  invisible(NULL)
}

# =============================================================================
# REPORT EXPORT
# =============================================================================

#' Export analysis report to JSON and Excel
#'
#' Saves the results of a stability analysis to a JSON file and an Excel workbook.
#'
#' @param data Numeric vector of data.
#' @param stable_params List of estimated stable parameters.
#' @param qcv QCV statistic.
#' @param skew_kurt List with skewness and kurtosis.
#' @param normality List of normality test results.
#' @param verdict Final verdict string.
#' @param filename Base name for output files.
#' @return Invisibly returns a list containing the paths of the exported JSON and Excel files.
#' @importFrom jsonlite toJSON
#' @importFrom openxlsx write.xlsx
#' @export
export_analysis_report <- function(data, stable_params, qcv, skew_kurt, normality, verdict, filename = "stable_report") {
  summary_data <- list(
    n = length(data), skewness = skew_kurt$skewness,
    kurtosis = skew_kurt$kurtosis, qcv = qcv, verdict = verdict
  )

  report <- list(
    summary = summary_data,
    normality_tests = normality,
    stable_params = stable_params
  )

  json_str <- jsonlite::toJSON(report, pretty = TRUE, auto_unbox = TRUE)
  writeLines(json_str, paste0(filename, ".json"))

  tryCatch({
    df_summary <- data.frame(metric = names(summary_data),
                             value = unlist(summary_data), stringsAsFactors = FALSE)
    df_params <- data.frame(parameter = names(stable_params),
                            value = unlist(stable_params), stringsAsFactors = FALSE)
    openxlsx::write.xlsx(list(Summary = df_summary, Stable_Params = df_params),
                         file = paste0(filename, ".xlsx"))
  }, error = function(e) {
    warning("Could not export Excel file: ", e$message)
  })
  invisible(NULL)
}

# =============================================================================
# FULL ANALYSIS PIPELINE
# =============================================================================


#' Perform full stability analysis and export results
#'
#' Executes the full pipeline for stability analysis: normality tests, QCV computation,
#' parameter estimation, plotting, and report export.
#'
#' @param x Numeric vector of data.
#' @param filename Base name for output files (JSON and Excel). Written to tempdir().
#' @param qcv_threshold Threshold for QCV to consider distribution heavy-tailed.
#' @param fig_path Optional path to save the plot (PNG). If NULL, uses tempdir().
#' @param verbose Logical; if TRUE, prints progress messages. Default is FALSE.
#' @return Invisibly returns a character string containing a Markdown-formatted summary report.
#' @export
analyse_stable_distribution <- function(x, filename = "interval_analysis",
                                        qcv_threshold = 1.8,
                                        fig_path = NULL,
                                        verbose = FALSE) {
  x <- x[!is.na(x)]
  if (length(x) < 4) stop("Data must have at least 4 non-missing values for analysis.")

  if (is.null(fig_path)) {
    fig_path <- file.path(tempdir(), "stability_test_plot.png")
  }
  filename <- file.path(tempdir(), filename)

  result <- tryCatch({
    if (verbose) message("Running normality tests...")
    normal <- test_normality(x)

    if (verbose) message("Computing skewness, kurtosis, and QCV...")
    sk_kurt <- skew_kurtosis(x)
    qcv <- qcv_stat(x)

    if (verbose) message("Estimating stable distribution parameters...")
    stable_params <- estimate_stable_r(x)

    verdict <- if ((normal$Shapiro$pvalue < 0.05 || normal$Anderson$pvalue < 0.05) && qcv > qcv_threshold) {
      "*Distribution probablement alpha-stable* (non normale, queue lourde)"
    } else {
      "**Distribution probablement pas alpha-stable**"
    }

    report <- paste0(
      "## Stability Analysis Results\n\n",
      "### Dataset Summary\n",
      "- **Sample size:** ", length(x), "\n",
      "- **Skewness:** ", sprintf("%.3f", sk_kurt$skewness), "\n",
      "- **Kurtosis:** ", sprintf("%.3f", sk_kurt$kurtosis), "\n",
      "- **QCV Statistic:** ", sprintf("%.3f", qcv), "\n\n",
      "### Normality Tests\n\n",
      "#### Shapiro-Wilk Test\n",
      "- **Statistic:** ", sprintf("%.4f", normal$Shapiro$statistic), "\n",
      "- **p-value:** ", sprintf("%.4f", normal$Shapiro$pvalue), "\n",
      "- **Result:** ", ifelse(normal$Shapiro$pvalue >= 0.05, "Normal", "Non-normal"), "\n\n",
      "#### Anderson-Darling Test\n",
      "- **Statistic:** ", sprintf("%.4f", normal$Anderson$statistic), "\n",
      "- **p-value:** ", sprintf("%.4f", normal$Anderson$pvalue), "\n",
      "- **Result:** ", ifelse(normal$Anderson$pvalue >= 0.05, "Normal", "Non-normal"), "\n\n",
      "### Estimated Stable Parameters\n",
      "- **alpha (tail):** ", sprintf("%.3f", stable_params$alpha), "\n",
      "- **beta (skewness):** ", sprintf("%.3f", stable_params$beta), "\n",
      "- **gamma (scale):** ", sprintf("%.3f", stable_params$gamma), "\n",
      "- **delta (location):** ", sprintf("%.3f", stable_params$delta), "\n\n",
      "### Final Verdict\n\n", verdict
    )

    if (verbose) message("Generating diagnostic plot...")
    plot_vs_normal_stable(x, stable_params, fig_path = fig_path)

    if (verbose) message("Exporting analysis report...")
    export_analysis_report(x, stable_params, qcv, sk_kurt, normal, verdict, filename = filename)

    report
  }, error = function(e) {
    if (verbose) message("[analyse_stable_distribution] Error: ", e$message)
    paste("**Error in stability analysis:**", e$message)
  })

  return(invisible(result))
}

