% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LEdecomp.R
\name{LEdecomp}
\alias{LEdecomp}
\title{Function for applying different Life-Expectancy decomposition and sensitivity methods}
\usage{
LEdecomp(
  mx1,
  mx2,
  age = NULL,
  nx = NULL,
  n_causes = NULL,
  cause_names = NULL,
  sex1 = "t",
  sex2 = sex1,
  method = c("lifetable", "arriaga", "arriaga_sym", "sen_arriaga", "sen_arriaga_sym",
    "sen_arriaga_inst", "sen_arriaga_inst2", "sen_arriaga_sym_inst",
    "sen_arriaga_sym_inst2", "chandrasekaran_ii", "sen_chandrasekaran_ii",
    "sen_chandrasekaran_ii_inst", "sen_chandrasekaran_ii_inst2", "chandrasekaran_iii",
    "sen_chandrasekaran_iii", "sen_chandrasekaran_iii_inst",
    "sen_chandrasekaran_iii_inst2", "lopez_ruzicka", "lopez_ruzicka_sym",
    "sen_lopez_ruzicka", "sen_lopez_ruzicka_sym", "sen_lopez_ruzicka_inst",
    "sen_lopez_ruzicka_inst2", 
     "horiuchi", "stepwise", "numerical"),
  closeout = TRUE,
  opt = TRUE,
  tol = 1e-10,
  Num_Intervals = 20,
  symmetrical = TRUE,
  direction = "both",
  perturb = 1e-06,
  ...
)
}
\arguments{
\item{mx1}{numeric. Age-structured mortality rates for population 1 (vector, matrix, or data.frame). See Details section for more info.}

\item{mx2}{numeric. Age-structured mortality rates for population 2 (same shape as \code{mx1}).}

\item{age}{integer. Lower bound of each age group. If \code{NULL}, it will be inferred from data (see Details).}

\item{nx}{integer vector of age intervals (defaults to 1 when missing).}

\item{n_causes}{integer or \code{NULL}. If provided with stacked vectors, forces the number of causes (columns).}

\item{cause_names}{optional character vector of length \code{n_causes} giving labels for causes. Alternatively detected from \code{colnames(mx1)} in case given as a \code{matrix} or \code{data.frame}}

\item{sex1}{character. \code{"m"},\code{"f"}, or \code{"t"}, affects a0 treatment.}

\item{sex2}{character. \code{"m"},\code{"f"}, or \code{"t"}, affects a0 treatment.}

\item{method}{character. One of the methods in \code{method_registry$method}.}

\item{closeout}{logical. Close out at top age (\code{TRUE}) or assume closed final age group (\code{FALSE}).}

\item{opt}{logical. For lifetable, numerical, and instantaneous sensitivity-based methods, optimize rate averaging
to eliminate the decomposition residual?}

\item{tol}{numeric. Tolerance for rate-averaging optimization.}

\item{Num_Intervals}{integer. For methods that discretize an integral (e.g., Horiuchi).}

\item{symmetrical}{logical. For stepwise replacement only: average 1 to 2 and 2 to 1?}

\item{direction}{character. For stepwise replacement: "up", "down", or "both".}

\item{perturb}{numeric. Small perturbation for numerical derivatives.}

\item{...}{optional arguments passed to \code{numDeriv::grad()} or other internals

#' @details
Input dimensions are flexible to accommodate different coding styles and data layouts:

\strong{Accepted forms of \code{mx1} and \code{mx2}:}
\itemize{
\item \strong{Vector:} A single all-cause mortality schedule, one value per age.
In this case \code{age} must be the same length, or can be omitted and will default to
\code{0:(length(mx1)-1)}, unless we detect you might be using abridged age groups.
\item \strong{Matrix:} Rows represent ages, columns represent causes of death.
Row names, if any, and if numeric, are interpreted as ages and override the supplied \code{age} argument or our inferences. Column names are retained in the output for clarity.
\item \strong{Data frame (wide):} Same layout as a matrix, with an optional column named \code{age}.
\item \strong{Stacked vector:} A long, concatenated vector representing causes stacked on top of
each other (i.e., all ages for cause 1, then all ages for cause 2, and so on).
If you don't specify age, we try to detect age and the number of causes. But please specify age- it could be stacked also, or not! For example, when used inside a tidy pipeline you might write \code{mutate(LEdecomp(mx1, mx2, age))} where \code{age} is repeated for each cause, i.e. the code might look the same as if you were dealing with all-cause data. But in that case be careful data are ordered consistently.
}

\strong{Age detection and inference:}
\itemize{
\item If \code{age} is supplied explicitly, it is used as given.
\item If missing, \code{LEdecomp()} attempts to infer it from (in order):
row names, names of the input vector, a column named \code{"age"} in a data frame,
or heuristics for single-year (0:100) or abridged (0,1,5,10,…) schedules.
\item If \code{age} is repeated (e.g., \code{c(0:100, 0:100, 0:100)}), the function assumes
a stacked structure and collapses \code{age} to its unique sorted values.
The number of repetitions becomes \code{n_causes}.
}

\strong{Return shape:}

The output mirrors the input form:
\itemize{
\item If the inputs were vectors, outputs are vectors.
\item If inputs were matrices or data frames (wide), outputs are matrices.
\item If inputs were stacked vectors, outputs are stacked vectors in the same order.
}}
}
\value{
An object of class \code{"LEdecomp"}:
\itemize{
\item \code{mx1}, \code{mx2}, \code{age}, \code{sex1}, \code{sex2}, \code{method}, \code{closeout}, \code{opt}, \code{tol}, \code{Num_Intervals},
\code{symmetrical}, \code{direction}, \code{perturb}
\item \code{sens}: vector/matrix of sensitivities (same dimensions as inputs)
\item \code{LE1}, \code{LE2}: life expectancy for \code{mx1} and \code{mx2}
\item \code{LEdecomp}: vector/matrix of contributions (same shape as inputs)
}
}
\description{
A variety of exact or asymptotically exact life expectancy decomposition methods are implemented. Also, several life-expectancy decomposition sensitivity methods are implemented to answer how each age will change with an increase/decrease in life expectancy. See the package README and references for details.
}
\examples{
## Simple reproducible setup
set.seed(123)
a <- 0.001
b <- 0.07

## 1) Vector (single cause), single-year ages
age <- 0:50
mx1 <- a * exp(age * b)
mx2 <- (a / 2) * exp(age * b)

res_vec <- LEdecomp(
  mx1 = mx1, mx2 = mx2,
  age = age, nx = rep(1, length(age)),
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
round(sum(res_vec$LEdecomp), 4)

## 2) Matrix (multiple causes): rows = age, cols = causes
##    Build 3 causes with random positive weights per age
k <- 3
w1 <- matrix(runif(length(age) * k, 0.9, 1.1), nrow = length(age)); w1 <- w1 / rowSums(w1)
w2 <- matrix(runif(length(age) * k, 0.9, 1.1), nrow = length(age)); w2 <- w2 / rowSums(w2)
mx1_mat <- (mx1) * w1
mx2_mat <- (mx2) * w2
colnames(mx1_mat) <- colnames(mx2_mat) <- paste0("c", 1:k)
rownames(mx1_mat) <- rownames(mx2_mat) <- as.character(age)

res_mat <- LEdecomp(
  mx1 = mx1_mat, mx2 = mx2_mat,
  age = age, nx = rep(1, length(age)),
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
## Check: row-summed cause contributions equal all-cause result
res_all <- LEdecomp(
  mx1 = mx1, mx2 = mx2,
  age = age, nx = rep(1, length(age)),
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
all.equal(rowSums(res_mat$LEdecomp), res_all$LEdecomp, tolerance = 1e-7)

## 3) Data frame (wide): same as matrix but with an 'age' column
df1 <- data.frame(age = age, mx1_mat, check.names = FALSE)
df2 <- data.frame(age = age, mx2_mat, check.names = FALSE)
res_df <- LEdecomp(
  mx1 = df1, mx2 = df2,
  age = NULL, nx = rep(1, length(age)),
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
all.equal(res_df$LEdecomp, res_mat$LEdecomp, tolerance = 1e-8)

## 4) Stacked vector (long/concatenated): all ages for cause 1, then cause 2, etc.
##    If 'age' is repeated per cause, LEdecomp infers n_causes and collapses age.
mx1_stack <- as.vector(mx1_mat)  # column-major flattening
mx2_stack <- as.vector(mx2_mat)
age_rep   <- rep(age, k)         # typical tidy pipeline: age repeated per cause

res_stack <- LEdecomp(
  mx1 = mx1_stack, mx2 = mx2_stack,
  age = age_rep, nx = NULL,
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
## Output is a stacked vector matching the matrix baseline when flattened
all.equal(res_stack$LEdecomp, c(res_mat$LEdecomp), tolerance = 1e-8)

## 5) Abridged ages (0,1,5,10,...,110): inference when labels are missing
age_ab <- c(0L, 1L, seq.int(5L, 110L, by = 5L))
nx_ab  <- c(diff(age_ab), tail(diff(age_ab), 1L))
mx1_ab <- a * exp(age_ab * b)
mx2_ab <- (a / 2) * exp(age_ab * b)

## Explicit abridged example
res_ab_explicit <- LEdecomp(
  mx1 = mx1_ab, mx2 = mx2_ab,
  age = age_ab, nx = nx_ab,
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)

## Unlabeled abridged vector of the same length: age and nx inferred
res_ab_infer <- LEdecomp(
  mx1 = mx1_ab, mx2 = mx2_ab,
  age = NULL, nx = NULL,
  sex1 = "t", method = "sen_arriaga", opt = TRUE
)
all.equal(res_ab_infer$age, as.numeric(age_ab))
all.equal(res_ab_infer$LEdecomp, res_ab_explicit$LEdecomp, tolerance = 1e-8)

## 6) Rownames override age when they look like ages
##    Here we give the wrong 'age' but set rownames to "0","1",...,"50".
wrong_age <- age + 10
mx1_rn <- mx1_mat; mx2_rn <- mx2_mat
rownames(mx1_rn) <- rownames(mx2_rn) <- as.character(age)
res_rn <- suppressWarnings(LEdecomp(
  mx1 = mx1_rn, mx2 = mx2_rn,
  age = wrong_age, nx = rep(1, length(age)),
  sex1 = "t", method = "sen_arriaga", opt = TRUE
))
all.equal(res_rn$age, as.numeric(age))

## 7) List available methods
available_methods()
}
\references{
\insertRef{arriaga1984measuring}{LEdecomp}
\insertRef{Chandrasekaran1986}{LEdecomp}
\insertRef{preston2000demography}{LEdecomp}
\insertRef{Ponnapalli2005}{LEdecomp}
\insertRef{horiuchi2008decomposition}{LEdecomp}
\insertRef{andreev2002algorithm}{LEdecomp}
}
\seealso{
\code{\link[=sen_e0_mx_lt]{sen_e0_mx_lt()}}, \code{\link[=arriaga]{arriaga()}}, \code{\link[=arriaga_sym]{arriaga_sym()}},
\code{\link[=sen_arriaga]{sen_arriaga()}}, \code{\link[=sen_arriaga_sym]{sen_arriaga_sym()}}
}
