#' @title S3 Methods: print
#'
#' @description
#' Provides user-friendly, formatted console output for objects generated by the \code{LCPA} package.
#' This generic function dispatches to class-specific methods that display concise summaries of model results,
#' simulated datasets, fit indices, model comparisons, and standard errors. Designed for interactive use
#' and quick diagnostic inspection.
#'
#' @param x An object of one of the following classes:
#'   \itemize{
#'     \item Model objects: \code{\link[LCPA]{LCA}}, \code{\link[LCPA]{LPA}},
#'       \code{\link[LCPA]{LCPA}}, \code{\link[LCPA]{LTA}}
#'     \item Simulation objects: \code{\link[LCPA]{sim.LCA}}, \code{\link[LCPA]{sim.LPA}},
#'       \code{\link[LCPA]{sim.LTA}}
#'     \item Fit/comparison objects: \code{\link[LCPA]{get.fit.index}}, \code{\link[LCPA]{compare.model}}
#'     \item Standard error objects: \code{\link[LCPA]{get.SE}}
#'     \item Summary objects: \code{summary.LCA}, \code{summary.LPA}, \code{summary.LCPA},
#'       \code{summary.LTA}, \code{summary.sim.LCA}, \code{summary.sim.LPA}, \code{summary.sim.LTA},
#'       \code{summary.fit.index}, \code{summary.compare.model}, \code{summary.SE}
#'   }
#' @param ... Additional arguments passed to methods (currently ignored in most cases).
#' @param digits Number of decimal places for numeric output (default: varies by method, often 4).
#'   Used by: \code{print.SE}, \code{print.summary.fit.index}, \code{print.summary.sim.LCA/LPA/LTA},
#'   \code{print.summary.SE}.
#' @param I.max Maximum number of variables/items to display before truncation (default: varies, e.g., 5).
#'   Used by: \code{print.SE}, \code{print.summary.sim.LCA/LPA/LTA}.
#' @param L.max Maximum number of latent classes/profiles to display before truncation (default: varies, e.g., 3).
#'   Used by: \code{print.SE}, \code{print.summary.sim.LTA}.
#'
#' @return Invisibly returns the input object \code{x}. No data is modified.
#'
#' @details
#' Each method produces a structured, human-readable summary optimized for its object type:
#'
#' \describe{
#'   \item{\strong{Model Objects (\code{LCA}/\code{LPA}/\code{LCPA}/\code{LTA})}}{
#'     Invokes \code{summary()} internally and prints comprehensive output including:
#'     \itemize{
#'       \item Model call and configuration (method, constraints, reference class)
#'       \item Data characteristics (N, I, time points, distribution)
#'       \item Fit statistics (LogLik, AIC, BIC, entropy, npar)
#'       \item Class/profile prior probabilities and frequencies
#'       \item Item-response probabilities (\code{LCA}) or profile means (\code{LPA})
#'       \item For \code{LCPA}/\code{LTA}: regression coefficients with significance markers and 95% CIs
#'       \item Convergence diagnostics (iterations, tolerance, hardware)
#'       \item Replication details (if \code{nrep > 1})
#'     }
#'   }
#'
#'   \item{\strong{Simulation Objects (\code{sim.LCA}/\code{sim.LPA}/\code{sim.LTA})}}{
#'     Displays simulation design and true parameter structure:
#'     \itemize{
#'       \item Configuration (N, I, L, times, constraint, distribution)
#'       \item True class/profile proportions and observed frequencies
#'       \item For \code{sim.LCA}: item category structure and conditional probabilities
#'       \item For \code{sim.LPA}: covariance constraint description and mean ranges
#'       \item For \code{sim.LTA}: transition mode (fixed/covariate), initial/transition coefficients
#'     }
#'     Output is truncated for high-dimensional structures using \code{I.max} and \code{L.max}.
#'   }
#'
#'   \item{\strong{Fit Index Objects (\code{fit.index})}}{
#'     Presents a clean table of model fit criteria:
#'     \itemize{
#'       \item Header with dimensions (N, I, L, npar)
#'       \item Formatted table: AIC, BIC, SABIC, CAIC, AWE, -2LL, SIC
#'       \item Interpretation note: “Lower values preferred for ICs”
#'       \item Values rounded to \code{digits} decimal places
#'     }
#'   }
#'
#'   \item{\strong{Model Comparison Objects (\code{compare.model})}}{
#'     Compares two nested models with statistical tests:
#'     \itemize{
#'       \item Comparative fit table (npar, LogLik, AIC, BIC, entropy)
#'       \item Classification quality (AvePP per class, overall entropy)
#'       \item Bayes Factor with interpretive guidance
#'       \item Likelihood ratio tests (standard, VLMR, Bootstrap) with p-values and significance codes
#'       \item Clear section headers and visual separators
#'     }
#'   }
#'
#'   \item{\strong{Standard Error Objects (\code{SE})}}{
#'     Displays uncertainty estimates for model parameters:
#'     \itemize{
#'       \item Class probability SEs (always fully shown)
#'       \item Profile means SEs (\code{LPA}) or item-response SEs (\code{LCA}), truncated by \code{L.max}/\code{I.max}
#'       \item Covariance SE summary (non-zero count only; full access via \code{extract()})
#'       \item Diagnostics: Bootstrap completion % or Hessian condition number with stability warnings
#'     }
#'   }
#'
#'   \item{\strong{Summary Objects}}{
#'     All \code{summary.*} methods are called internally by their corresponding \code{print.*} methods.
#'     They pre-compute and structure output for consistent formatting. Direct calls are also supported.
#'   }
#' }
#'
#' @section Output Conventions:
#' \itemize{
#'   \item Numeric values are typically rounded to 4 decimal places unless overridden by \code{digits}.
#'   \item Large matrices (e.g., item parameters, transition coefficients) are truncated with clear messages.
#'   \item Significance markers: \code{***} (<0.001), \code{**} (<0.01), \code{*} (<0.05), \code{.} (<0.1).
#'   \item 95% confidence intervals computed as: Estimate ± 1.96 × Std_Error.
#'   \item Reference classes (for multinomial models) are explicitly stated.
#'   \item Warnings appear for unstable SEs (high condition number) or incomplete Bootstrap runs.
#' }
#'
#' @name print
NULL

#' @describeIn print Print method for \code{LCA} objects
#' @export
print.LCA <- function(x, ...) {
  print.summary.LCA(summary(x))
  invisible(x)
}

#' @describeIn print Print method for \code{summary.LCA} objects
#' @export
print.summary.LCA <- function(x, ...) {
  digits <- x$digits

  # Print package info header
  printPackageInfo()
  cat("==============================================\n")
  cat("LATENT CLASS ANALYSIS (LCA) MODEL SUMMARY\n")
  cat("==============================================\n\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Model Configuration
  cat("\nLatent Class Analysis Model\n")
  cat(paste("Latent Classes:", x$model.config$latent_classes, "\n"))
  cat(paste("Estimation Method:", x$model.config$method, "\n"))

  # Data Characteristics
  cat("\nData Characteristics:\n")
  cat(sprintf("  Sample Size: %d observations\n", x$data.info$N))
  cat(sprintf("  Items: %d\n", x$data.info$I))
  if (x$data.info$uniform_categories) {
    cat(sprintf("  Response Categories per Item: %d (uniform)\n", x$data.info$poly.value[1]))
  } else {
    cat("  Response Categories per Item:\n")
    cat(paste("    ", paste(sprintf("Item %d: %d categories",
                                    seq_along(x$data.info$poly.value), x$data.info$poly.value),
                            collapse = "\n    "),
              "\n", sep = ""))
  }

  # Fit Statistics
  cat("\nFit Statistics:\n")
  cat(sprintf("  Log-likelihood: %.*f\n", digits, x$fit.stats$LogLik))
  cat(sprintf("  AIC: %.*f\n", digits, x$fit.stats$AIC))
  cat(sprintf("  BIC: %.*f\n", digits, x$fit.stats$BIC))
  cat(sprintf("  Entropy: %.*f (classification quality: 0=worst, 1=best)\n", digits, x$fit.stats$entropy))
  cat(sprintf("  Free Parameters: %d\n", x$fit.stats$npar))

  # Class Probabilities
  cat("\nClass Prior Probabilities:\n")
  print(format(x$class.probs, digits = digits), row.names = FALSE)

  # Item Probabilities
  if (x$total.items > 0) {
    cat("\nItem-Response Probabilities (", x$I.max.shown, "items ):\n")
    for (i in seq_along(x$item.probs)) {
      cat(sprintf("\nItem %d:\n", i))
      print(x$item.probs[[i]], digits = digits)
    }
    if (x$total.items > x$I.max.shown) {
      cat(sprintf("\n[Only first %d items shown. Access full via original object$probability]\n",
                  x$I.max.shown))
    }
  }

  # Convergence
  if (length(x$convergence) > 0) {
    cat("\nConvergence Details:\n")
    conv <- x$convergence
    cat(sprintf("  Algorithm: %s\n", conv$algorithm))
    if(!is.null(conv$iterations)){
      cat(sprintf("  Converged after %d iterations\n", conv$iterations))
    }
    if(!is.null(conv$note)){
      cat(sprintf("  Note\n", conv$note))
    }
    if (!is.null(conv$tolerance)) {
      cat(sprintf("  Convergence tolerance: %.1e\n", conv$tolerance))
    }
    if (!is.null(conv$early_stop_threshold)) {
      cat(sprintf("  Early stopping threshold: %d iterations\n", conv$early_stop_threshold))
    }
    if(!is.null(conv$loglik_change)){
      cat(sprintf("  Log-likelihood change: |%.2f - %.2f| = %.4f\n",
                  conv$loglik_initial, conv$loglik_final, conv$loglik_change))
    }
    if(!is.null(conv$hardware)){
      cat(sprintf("  The LCA is run on %s\n", conv$hardware))
    }
  }

  # Replication
  if (!is.null(x$replication)) {
    cat("\nReplication Details:\n")
    cat(sprintf("  Completed replications: %d\n", x$replication$nrep))
    cat(sprintf("  Best BIC solution selected (BIC = %.2f)\n", x$replication$best_BIC))
  }

  invisible(x)
}

#' @describeIn print Print method for \code{LPA} objects
#' @export
print.LPA <- function(x, ...) {
  print.summary.LPA(summary(x))
  invisible(x)
}

#' @describeIn print Print method for \code{summary.LPA} objects
#' @export
print.summary.LPA <- function(x, ...) {
  digits <- x$digits

  # Print header
  printPackageInfo()
  cat("==============================================\n")
  cat("LATENT PROFILE ANALYSIS (LPA) MODEL SUMMARY\n")
  cat("==============================================\n\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Model Configuration
  cat("\nModel Configuration:\n")
  cat(paste("  Latent Profiles:", x$model.config$latent_profiles, "\n"))
  cat(paste("  Structure:\n", x$model.config$cov_structure, "\n"))
  cat(paste("  Estimation Method:", x$model.config$method, "\n"))

  # Data Characteristics
  cat("\nData Characteristics:\n")
  cat(sprintf("  Sample Size: %d observations\n", x$data.info$N))
  cat(sprintf("  Variables: %d continuous variables\n", x$data.info$I))
  cat(sprintf("  Distributional Assumption: %s\n", x$data.info$distribution))

  # Fit Statistics
  cat("\nFit Statistics:\n")
  cat(sprintf("  Log-likelihood: %.*f\n", digits, x$fit.stats$LogLik))
  cat(sprintf("  AIC: %.*f\n", digits, x$fit.stats$AIC))
  cat(sprintf("  BIC: %.*f\n", digits, x$fit.stats$BIC))
  cat(sprintf("  Entropy: %.*f (classification quality: 0=worst, 1=best)\n", digits, x$fit.stats$entropy))
  cat(sprintf("  Free Parameters: %d\n", x$fit.stats$npar))

  # Class Probabilities
  cat("\nProfile Prior Probabilities:\n")
  print(format(x$class.probs, digits = digits), row.names = FALSE, print.gap = 2)

  # Profile Means
  if (x$total.vars > 0) {
    cat("\nProfile Means (", x$I.max.shown, "variables):\n")
    print(x$class.means, digits = digits, na.print = "NA")

    if (x$total.vars > x$I.max.shown) {
      cat(sprintf("\n[Only first %d variables shown. Access full means via original object$params$means]\n",
                  x$I.max.shown))
    }
  }

  # Convergence
  if (length(x$convergence) > 0) {
    cat("\nConvergence Details:\n")
    conv <- x$convergence
    cat(sprintf("  Algorithm: %s\n", conv$algorithm))

    if (!is.null(conv$note)) {
      cat(sprintf("  Note: %s\n", conv$note))
    }
    if (!is.null(conv$iterations)) {
      cat(sprintf("  Converged after %d iterations\n", conv$iterations))
      if (!is.null(conv$tolerance)) {
        cat(sprintf("  Convergence tolerance: %.1e\n", conv$tolerance))
      }
      if (!is.null(conv$early_stop_threshold)) {
        cat(sprintf("  Early stopping threshold: %d iterations\n", conv$early_stop_threshold))
      }
      cat(sprintf("  Log-likelihood change: |%.2f - %.2f| = %.4f\n",
                  conv$loglik_initial, conv$loglik_final, conv$loglik_change))
    }
    if(!is.null(conv$hardware)){
      cat(sprintf("  The LPA is run on %s\n", conv$hardware))
    }
  }

  # Replication
  if (!is.null(x$replication)) {
    cat("\nReplication Details:\n")
    cat(sprintf("  Completed replications: %d\n", x$replication$nrep))
    cat(sprintf("  Best BIC solution selected (BIC = %.2f)\n", x$replication$best_BIC))
  }

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{LTA} objects
#' @export
print.LTA <- function(x, ...) {
  print.summary.LTA(summary(x))
  invisible(x)
}

#' @describeIn print Print method for \code{summary.LTA} objects
#' @export
print.summary.LTA <- function(x, ...) {
  digits <- x$digits

  # Print package info and title
  cat("==============================================\n")
  cat("LATENT TRANSITION ANALYSIS SUMMARY\n")
  cat("==============================================\n\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Model Configuration
  cat("\nModel Configuration:\n")
  cat(sprintf("  Time Points: %d\n", x$model.config$time_points))
  cat(sprintf("  Latent Classes: %d\n", x$model.config$latent_classes))
  cat(sprintf("  Model Type: %s\n", x$model.config$model_type))
  cat(sprintf("  Reference Class: %d\n", x$model.config$reference_class))
  cat(sprintf("  Covariates Mode: %s\n", x$model.config$covariates_mode))
  cat(sprintf("  CEP Handling: %s\n", x$model.config$CEP_handling))
  cat(sprintf("  Transition Mode: %s\n", x$model.config$transition_mode))

  # Data Information
  cat("\nData Information:\n")
  cat(sprintf("  Sample Size: %d\n", x$data.info$sample_size))
  cat(sprintf("  Variables: %d\n", x$data.info$variables))
  cat(sprintf("  Time Points: %d\n", x$data.info$time_points))

  # Fit Statistics
  cat("\nFit Statistics:\n")
  cat(sprintf("  Log-Likelihood: %.*f\n", digits, x$fit.stats$LogLik))
  cat(sprintf("  AIC: %.*f\n", digits, x$fit.stats$AIC))
  cat(sprintf("  BIC: %.*f\n", digits, x$fit.stats$BIC))
  cat(sprintf("  Free Parameters: %d\n", x$fit.stats$npar))

  # Class Probabilities Over Time
  cat("\nClass Probabilities Over Time:\n")
  for (t in 1:length(x$class.probs)) {
    cat(sprintf("\nTime Point %d:\n", t))
    probs_df <- x$class.probs[[t]]
    probs_df$Probability <- format(as.numeric(probs_df$Probability), digits = digits)
    print(probs_df[, c("Class", "Probability", "Proportion", "Frequency")],
          row.names = FALSE, right = TRUE)
  }

  # Initial State Model
  cat("\nInitial State Model (Time 1):\n")
  cat(sprintf("  Reference Class: Class %d (coefficients fixed to zero)\n", x$initial_model$reference_class))
  if (nrow(x$initial_model$coefficients) > 0) {
    print_coef_table(x$initial_model$coefficients, digits)
  } else {
    cat("  No covariate effects on initial state\n")
  }

  # Transition Models - CORRECTED EXPLANATION
  cat("\nTransition Models:\n")
  cat(sprintf("  Reference Destination Class: Class %d (all transitions are relative to this class)\n", x$reference_class))

  if (x$covariates.timeCross) {
    cat("  Time-invariant transition effects (coefficients constant across time):\n")
    if (nrow(x$transition_models[[1]]) > 0) {
      cat("\nRepresentative Transition (Time 1 -> Time 2):\n")
      print_transition_table(x$transition_models[[1]], digits)
      cat("\n[All time points share these same transition coefficients]\n")
    } else {
      cat("  No covariate effects on transitions\n")
    }
  } else {
    for (name in names(x$transition_models)) {
      trans_df <- x$transition_models[[name]]
      if (nrow(trans_df) > 0) {
        cat(sprintf("\n%s:\n", name))
        print_transition_table(trans_df, digits)
      } else {
        cat(sprintf("\n%s: No covariate effects on transitions\n", name))
      }
    }
  }

  # Convergence Information
  cat("\nConvergence Information:\n")
  cat(sprintf("  Coveraged: %s\n", x$convergence$coveraged))
  cat(sprintf("  Iterations: %d\n", x$convergence$iterations))
  cat(sprintf("  Status: %s\n", x$convergence$converg_note))

  cat("\n")
  invisible(x)
}

# Helper function to print coefficient tables with significance markers and 95% CI
print_coef_table <- function(df, digits) {
  if (nrow(df) == 0) return(cat("  No coefficients to display\n"))

  # Add significance markers
  if ("p_value" %in% names(df)) {
    df$Sig <- ""
    p_vals <- as.numeric(df$p_value)
    df$Sig[!is.na(p_vals) & p_vals < 0.001] <- "***"
    df$Sig[!is.na(p_vals) & p_vals < 0.01 & p_vals >= 0.001] <- "**"
    df$Sig[!is.na(p_vals) & p_vals < 0.05 & p_vals >= 0.01] <- "*"
    df$Sig[!is.na(p_vals) & p_vals < 0.1 & p_vals >= 0.05] <- "."
  }

  # Create 95% CI column
  if ("lower_95" %in% names(df) && "upper_95" %in% names(df)) {
    valid_ci <- !is.na(df$lower_95) & !is.na(df$upper_95) & is.finite(df$lower_95) & is.finite(df$upper_95)
    df$`95% CI` <- "NA"
    df$`95% CI`[valid_ci] <- sprintf("(%.*f, %.*f)", digits, df$lower_95[valid_ci], digits, df$upper_95[valid_ci])
  }

  # Format numeric columns
  if ("Estimate" %in% names(df)) {
    valid_est <- !is.na(df$Estimate) & is.finite(df$Estimate)
    df$Estimate <- as.character(df$Estimate)
    df$Estimate[valid_est] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$Estimate[valid_est]))
  }

  if ("Std_Error" %in% names(df)) {
    valid_se <- !is.na(df$Std_Error) & is.finite(df$Std_Error)
    df$Std_Error <- as.character(df$Std_Error)
    df$Std_Error[valid_se] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$Std_Error[valid_se]))
  }

  if ("z_value" %in% names(df)) {
    valid_z <- !is.na(df$z_value) & is.finite(df$z_value)
    df$z_value <- as.character(df$z_value)
    df$z_value[valid_z] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$z_value[valid_z]))
  }

  # Define column order with 95% CI after Std_Error
  cols <- c("Class", "Covariate", "Estimate", "Std_Error")
  if ("95% CI" %in% names(df)) cols <- c(cols, "95% CI")
  cols <- c(cols, "z_value")
  if ("Sig" %in% names(df)) cols <- c(cols, "Sig")

  print_df <- df[, cols, drop = FALSE]
  colnames(print_df) <- gsub("_", " ", colnames(print_df))

  print(print_df, row.names = FALSE, right = TRUE)
  if ("Sig" %in% names(df)) {
    cat("Signif. codes:  *** < 0.001, ** < 0.01, * < 0.05, . < 0.1\n")
  }
  cat("95% confidence intervals calculated as: Estimate +/- 1.96 * Std_Error\n")
}

# Helper function to print transition tables with significance markers and 95% CI
print_transition_table <- function(df, digits) {
  if (nrow(df) == 0) return(cat("  No coefficients to display\n"))

  # Add significance markers
  if ("p_value" %in% names(df)) {
    df$Sig <- ""
    p_vals <- as.numeric(df$p_value)
    df$Sig[!is.na(p_vals) & p_vals < 0.001] <- "***"
    df$Sig[!is.na(p_vals) & p_vals < 0.01 & p_vals >= 0.001] <- "**"
    df$Sig[!is.na(p_vals) & p_vals < 0.05 & p_vals >= 0.01] <- "*"
    df$Sig[!is.na(p_vals) & p_vals < 0.1 & p_vals >= 0.05] <- "."
  }

  # Create 95% CI column
  if ("lower_95" %in% names(df) && "upper_95" %in% names(df)) {
    valid_ci <- !is.na(df$lower_95) & !is.na(df$upper_95) & is.finite(df$lower_95) & is.finite(df$upper_95)
    df$`95% CI` <- "NA"
    df$`95% CI`[valid_ci] <- sprintf("(%.*f, %.*f)", digits, df$lower_95[valid_ci], digits, df$upper_95[valid_ci])
  }

  # Format numeric columns
  if ("Estimate" %in% names(df)) {
    valid_est <- !is.na(df$Estimate) & is.finite(df$Estimate)
    df$Estimate <- as.character(df$Estimate)
    df$Estimate[valid_est] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$Estimate[valid_est]))
  }

  if ("Std_Error" %in% names(df)) {
    valid_se <- !is.na(df$Std_Error) & is.finite(df$Std_Error)
    df$Std_Error <- as.character(df$Std_Error)
    df$Std_Error[valid_se] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$Std_Error[valid_se]))
  }

  if ("z_value" %in% names(df)) {
    valid_z <- !is.na(df$z_value) & is.finite(df$z_value)
    df$z_value <- as.character(df$z_value)
    df$z_value[valid_z] <- sprintf(paste0("%.", digits, "f"), as.numeric(df$z_value[valid_z]))
  }

  # Define column order with 95% CI after Std_Error
  cols <- c("From_Class", "To_Class", "Covariate", "Estimate", "Std_Error")
  if ("95% CI" %in% names(df)) cols <- c(cols, "95% CI")
  cols <- c(cols, "z_value")
  if ("Sig" %in% names(df)) cols <- c(cols, "Sig")

  print_df <- df[, cols, drop = FALSE]
  colnames(print_df) <- gsub("_", " ", colnames(print_df))

  print(print_df, row.names = FALSE, right = TRUE)
  if ("Sig" %in% names(df)) {
    cat("Signif. codes:  *** < 0.001, ** < 0.01, * < 0.05, . < 0.1\n")
  }
  cat("Interpretation: Coefficients represent log odds of transitioning to 'To Class'\n")
  cat("                relative to reference destination class (Class ")
  cat(attr(df, "ref_class_note", exact = TRUE))
  cat(")\n")
  cat("95% confidence intervals calculated as: Estimate +/- 1.96 * Std_Error\n")
}

#' @describeIn print Print method for \code{LCPA} objects
#' @export
print.LCPA <- function(x, ...) {
  print.summary.LCPA(summary(x))
  invisible(x)
}


#' @describeIn print Print method for \code{summary.LCPA} objects
#' @export
print.summary.LCPA <- function(x, ...) {
  digits <- x$digits

  # Print package info and title
  cat("================================================\n")
  cat("LATENT CLASS/PROFILE ANALYSIS SUMMARY (CROSS-SECTIONAL)\n")
  cat("================================================\n\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Model Configuration
  cat("\nModel Configuration:\n")
  cat(sprintf("  Latent Classes: %d\n", x$model.config$latent_classes))
  cat(sprintf("  Model Type: %s\n", x$model.config$model_type))
  cat(sprintf("  Reference Class: %d (coefficients fixed to zero)\n", x$model.config$reference_class))
  cat(sprintf("  Covariates Mode: %s\n", x$model.config$covariates_mode))
  cat(sprintf("  CEP Handling: %s\n", x$model.config$CEP_handling))

  # Data Information
  cat("\nData Information:\n")
  cat(sprintf("  Sample Size: %d\n", x$data.info$sample_size))
  cat(sprintf("  Variables: %d\n", x$data.info$variables))

  # Fit Statistics
  cat("\nFit Statistics:\n")
  cat(sprintf("  Log-Likelihood: %.*f\n", digits, x$fit.stats$LogLik))
  cat(sprintf("  AIC: %.*f\n", digits, x$fit.stats$AIC))
  cat(sprintf("  BIC: %.*f\n", digits, x$fit.stats$BIC))
  cat(sprintf("  Free Parameters: %d\n", x$fit.stats$npar))

  # Class Probabilities
  cat("\nClass Membership Probabilities:\n")
  probs_df <- x$class.probs
  probs_df$Probability <- format(as.numeric(probs_df$Probability), digits = digits)
  print(probs_df[, c("Class", "Probability", "Proportion", "Frequency")],
        row.names = FALSE, right = TRUE)

  # Coefficients Table
  cat("\nClass Membership Model (Multinomial Logit):\n")
  cat(sprintf("  Reference Class: Class %d (all coefficients relative to this class)\n", x$reference_class))

  if (nrow(x$coefficients) > 0) {
    print_coef_table(x$coefficients, digits)
  } else {
    cat("  No covariate effects estimated (intercept-only model)\n")
  }

  # Note if truncated
  if (x$total.vars > x$I.max.shown) {
    cat(sprintf("\nNote: Only top %d covariates shown (total: %d). Use summary(object, I.max = Inf) to see all.\n",
                x$I.max.shown, x$total.vars))
  }

  # Convergence Information
  cat("\nConvergence Information:\n")
  cat(sprintf("  Coveraged: %s\n", x$convergence$coveraged))
  cat(sprintf("  Iterations: %d\n", x$convergence$iterations))
  cat(sprintf("  Status: %s", x$convergence$converg_note))

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{sim.LCA} objects
#' @export
print.sim.LCA <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LATENT CLASS ANALYSIS DATA\n")
  cat("==============================================\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Basic configuration
  N <- nrow(x$response)
  I <- ncol(x$response)
  L <- length(x$P.Z)
  uniform_cat <- length(unique(x$poly.value)) == 1

  cat(sprintf("\nConfiguration:\n  Sample Size: %d\n  Variables: %d\n  Latent Classes: %d\n",
              N, I, L))

  if (uniform_cat) {
    cat(sprintf("  Categories per Variable: %d (uniform)\n", x$poly.value[1]))
  } else {
    cat("  Categories per Variable:\n")
    cat(paste("    ", paste(sprintf("Var %d: %d cats",
                                    seq_along(x$poly.value), x$poly.value),
                            collapse = "\n    "), "\n", sep = ""))
  }

  cat(sprintf("  Item Quality (IQ): %s\n",
              if(is.numeric(x$arguments$IQ)) sprintf("%.2f", x$arguments$IQ) else x$arguments$IQ))
  cat(sprintf("  Class Distribution: %s\n", x$arguments$distribution))

  # Class proportions
  cat("\nClass Proportions:\n")
  class_props <- data.frame(
    Class = paste0("L", 1:L),
    Proportion = sprintf("%.1f%%", round(x$P.Z * 100, 1)),
    Frequency = as.vector(table(x$Z))
  )
  print(class_props, row.names = FALSE)

  invisible(x)
}

#' @describeIn print Print method for \code{summary.sim.LCA} objects
#' @export
print.summary.sim.LCA <- function(x, ...) {
  digits <- x$digits

  # Title
  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LCA DATA SUMMARY\n")
  cat("==============================================\n\n")

  cat("Call:\n")
  print(x$call)

  cat("\nSimulation Configuration:\n")
  cat(sprintf("  Sample Size: %d observations\n", x$config$N))
  cat(sprintf("  Variables: %d\n", x$config$I))
  cat(sprintf("  Latent Classes: %d\n", x$config$L))
  cat(sprintf("  Item Quality (IQ): %s\n",
              if(is.numeric(x$config$IQ)) sprintf("%.3f", x$config$IQ) else x$config$IQ))
  cat(sprintf("  Class Distribution: %s\n", x$config$distribution))

  # Response structure
  if (x$config$uniform_categories) {
    cat(sprintf("\nResponse Structure:\n  Categories per Variable: %d (uniform)\n",
                x$config$poly.value[1]))
  } else {
    cat("\nResponse Structure:\n")
    cat("  Categories per Variable:\n")
    cat(paste("    ", paste(sprintf("Var %d: %d categories",
                                    seq_along(x$config$poly.value), x$config$poly.value),
                            collapse = "\n    "), "\n", sep = ""))
  }

  cat("\nTrue Class Proportions:\n")
  print(x$class.probs, row.names = FALSE)

  if (x$total.vars > 0) {
    cat(sprintf("\nTrue Conditional Probabilities (first %d variables):\n", x$I.max.shown))
    for (i in seq_along(x$item.probs)) {
      cat(sprintf("\nVariable %d:\n", i))
      print(x$item.probs[[i]], digits = digits)
    }
    if (x$total.vars > x$I.max.shown) {
      cat(sprintf("\n[Only first %d variables shown. Full probabilities in object$par]\n",
                  x$I.max.shown))
    }
  }

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{sim.LPA} objects
#' @export
print.sim.LPA <- function(x, ...) {
  # Print package info and title
  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LATENT PROFILE ANALYSIS DATA\n")
  cat("==============================================\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Extract basic info
  N <- nrow(x$response)
  I <- ncol(x$response)
  L <- length(x$P.Z)
  var_names <- colnames(x$response)

  # Configuration info
  cat("\nConfiguration:\n")
  cat(sprintf("  Sample Size: %d observations\n", N))
  cat(sprintf("  Variables: %d continuous variables\n", I))
  cat(sprintf("  Latent Profiles: %d\n", L))

  # Covariance constraint description
  constraint_desc <- describe_constraint(x$constraint, I)
  cat(sprintf("  Covariance Structure: %s\n", constraint_desc$description))

  # Class distribution
  cat(sprintf("  Class Distribution: %s\n", x$arguments$distribution))

  # Class proportions
  cat("\nTrue Profile Proportions:\n")
  class_props <- data.frame(
    Profile = names(x$P.Z),
    Proportion = sprintf("%.1f%%", x$P.Z * 100),
    Frequency = as.vector(table(x$Z))
  )
  print(class_props, row.names = FALSE, right = TRUE)

  # Mean range
  means_range <- range(x$means)
  cat(sprintf("\nMean Values Range: [%.2f, %.2f]\n", means_range[1], means_range[2]))

  invisible(x)
}

#' @describeIn print Print method for \code{summary.sim.LPA} objects
#' @export
print.summary.sim.LPA <- function(x, ...) {
  digits <- x$digits

  # Print package info and title
  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LPA DATA SUMMARY\n")
  cat("==============================================\n\n")

  # Configuration details
  cat("\nSimulation Configuration:\n")
  cat(sprintf("  Sample Size: %d observations\n", x$config$N))
  cat(sprintf("  Variables: %d continuous\n", x$config$I))
  cat(sprintf("  Latent Profiles: %d\n", x$config$L))
  cat(sprintf("  Class Distribution: %s\n", x$config$distribution))

  # Covariance structure
  cat("\nCovariance Structure:\n")
  cat(sprintf("  Type: %s\n", x$config$constraint_desc$description))
  if (!is.null(x$config$constraint_desc$details) && x$config$constraint_desc$details != "") {
    cat("  Details:\n")
    cat(paste0("    ", strwrap(x$config$constraint_desc$details, width = 60, indent = 4, exdent = 4), collapse = "\n"), "\n")
  }

  # Class proportions
  cat("\nTrue Profile Proportions:\n")
  formatted_probs <- sprintf(paste0("%.", digits, "f"), x$class.probs$Probability)
  class_props <- data.frame(
    Profile = x$class.probs$Profile,
    Probability = formatted_probs,
    Proportion = sprintf("%.1f%%", as.numeric(formatted_probs) * 100),
    Frequency = x$class.probs$Frequency
  )
  print(class_props, row.names = FALSE, right = TRUE)

  # Profile means
  if (x$total.vars > 0) {
    cat(sprintf("\nTrue Profile Means (first %d variables):\n", x$I.max.shown))
    formatted_means <- format(x$class.means, digits = digits, nsmall = digits, justify = "right")
    print(formatted_means, quote = FALSE, right = TRUE)

    if (x$total.vars > x$I.max.shown) {
      cat(sprintf("\n[Only first %d variables shown. Full means in object$means]\n",
                  x$I.max.shown))
    }
  }

  # Covariance matrix info
  if (x$config$I > 1) {
    cat("\nCovariance Matrix Information:\n")
    example_cov <- x$config$constraint_desc$example
    if (!is.null(example_cov)) {
      cat("  Example covariance matrices available in object$covs\n")
    } else {
      cat("  Full covariance matrices available in object$covs\n")
    }
  }

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{sim.LTA} objects
#' @export
print.sim.LTA <- function(x, ...) {
  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LATENT TRANSITION ANALYSIS DATA\n")
  cat("==============================================\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  # Basic configuration
  N <- nrow(x$responses[[1]])
  I <- ncol(x$responses[[1]])
  L <- length(x$P.Zs[[1]])
  times <- length(x$responses)
  type <- if (!is.null(x$par)) "LCA" else "LPA"

  cat("\nConfiguration:\n")
  cat(sprintf("  Sample Size: %d\n", N))
  cat(sprintf("  Variables/Items: %d\n", I))
  cat(sprintf("  Latent Classes/Profiles: %d\n", L))
  cat(sprintf("  Time Points: %d\n", times))
  cat(sprintf("  Model Type: %s\n", ifelse(type == "LCA",
                                           "Latent Class Analysis",
                                           "Latent Profile Analysis")))

  # Covariate mode check
  use_covariates <- !is.null(x$covariates) || !is.null(x$beta) || !is.null(x$gamma)
  cat(sprintf("  Transition Mode: %s\n", ifelse(use_covariates, "Covariate-dependent", "Fixed probabilities")))

  # Class proportions at each time point
  cat("\nClass/Profile Proportions by Time Point:\n")
  for (t in 1:times) {
    class_props <- data.frame(
      Class = paste0("L", 1:L),
      Proportion = sprintf("%.1f%%", round(x$P.Zs[[t]] * 100, 1)),
      Frequency = as.vector(table(x$Zs[[t]]))
    )
    cat(sprintf("\nTime %d:\n", t))
    print(class_props, row.names = FALSE)
  }

  invisible(x)
}


#' @describeIn print Print method for \code{summary.sim.LTA} objects
#' @export
print.summary.sim.LTA <- function(x, ...) {
  digits <- x$digits

  printPackageInfo()
  cat("==============================================\n")
  cat("SIMULATED LTA DATA SUMMARY\n")
  cat("==============================================\n\n")

  cat("Call:\n")
  print(x$call)

  cat("\nSimulation Configuration:\n")
  cat(sprintf("  Sample Size: %d observations\n", x$config$N))
  cat(sprintf("  Variables/Items: %d\n", x$config$I))
  cat(sprintf("  Latent Classes/Profiles: %d\n", x$config$L))
  cat(sprintf("  Time Points: %d\n", x$config$times))
  cat(sprintf("  Model Type: %s\n", ifelse(x$config$type == "LCA",
                                           "Latent Class Analysis (categorical indicators)",
                                           "Latent Profile Analysis (continuous indicators)")))

  if (x$config$type == "LPA") {
    cat(sprintf("  Covariance Constraint: %s\n", x$config$constraint))
  }

  cat(sprintf("  Class Distribution: %s\n", x$config$distribution))

  # Class probabilities
  cat("\nTrue Class/Profile Proportions:\n")
  for (t in 1:length(x$class.probs)) {
    cat(sprintf("\nTime Point %d:\n", t))
    probs_df <- x$class.probs[[t]]
    probs_df$Probability <- sprintf(paste0("%.", digits, "f"), probs_df$Probability)
    probs_df$Proportion <- sprintf("%.1f%%", as.numeric(probs_df$Probability) * 100)
    print(probs_df[, c("Class", "Probability", "Proportion", "Frequency")],
          row.names = FALSE, right = TRUE)
  }

  # Parameters by model type
  if (x$config$type == "LCA") {
    cat(sprintf("\nTrue Conditional Probabilities (first %d variables at each time point):\n",
                x$I.max.shown))

    for (t in 1:length(x$item.probs)) {
      cat(sprintf("\nTime Point %d:\n", t))
      for (i in 1:x$I.max.shown) {
        cat(sprintf("\nVariable %d:\n", i))
        print(x$item.probs[[t]][[i]], digits = digits)
      }

      if (x$total.vars > x$I.max.shown) {
        cat(sprintf("\n[Only first %d variables shown for Time %d]\n",
                    x$I.max.shown, t))
      }
    }
  } else { # LPA
    cat(sprintf("\nTrue Class/Profile Means (first %d variables at each time point):\n",
                x$I.max.shown))

    for (t in 1:length(x$class.means)) {
      cat(sprintf("\nTime Point %d:\n", t))
      formatted_means <- format(x$class.means[[t]], digits = digits,
                                nsmall = digits, justify = "right")
      print(formatted_means, quote = FALSE, right = TRUE)

      if (x$total.vars > x$I.max.shown) {
        cat(sprintf("\n[Only first %d variables shown for Time %d]\n",
                    x$I.max.shown, t))
      }
    }
  }

  # Transition information
  cat("\nTransition Structure:\n")
  if (!is.null(x$transition) && !is.null(x$transition$mode)) {
    if (x$transition$mode == "fixed") {
      cat("  Mode: Fixed transition probabilities\n")

      # Get time points directly from stored data frame
      for (i in 1:nrow(x$transition$time_points)) {
        from_time <- x$transition$time_points$from[i]
        to_time <- x$transition$time_points$to[i]

        cat(sprintf("\nTransition Probabilities (Time %d -> Time %d):\n",
                    from_time, to_time))
        print(x$transition$rate[[i]], digits = digits)
      }
    } else if (x$transition$mode == "covariate") {
      cat("  Mode: Covariate-dependent transitions\n")

      # Initial state coefficients (beta)
      cat("\nInitial State Coefficients (beta):\n")
      print(x$transition$beta, digits = digits)

      # Transition coefficients (gamma)
      for (t in 1:length(x$transition$gamma)) {
        from_time <- x$transition$time_points$from[t]
        to_time <- x$transition$time_points$to[t]

        cat(sprintf("\nTransition Coefficients (Time %d -> Time %d):\n",
                    from_time, to_time))

        gamma_t <- x$transition$gamma[[t]]
        for (l in 1:min(x$L.max.shown, x$config$L)) {
          cat(sprintf("\nFrom Class %d:\n", l))
          print(gamma_t[[l]], digits = digits)
        }

        if (x$config$L > x$L.max.shown) {
          cat(sprintf("\n[Only first %d classes shown for this transition]\n",
                      x$L.max.shown))
        }
      }

      # Covariates information
      if (!is.null(x$covariates)) {
        cat("\nCovariates Summary of All Observetions:\n")
        for (t in 1:length(x$covariates)) {
          cat(sprintf("\nTime Point %d:\n", t))
          print(x$covariates[[t]], digits = digits, row.names = FALSE)
        }
      }
    }
  } else {
    cat("  No transition structure available\n")
  }

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{fit.index} objects
#' @export
print.fit.index <- function(x, ...) {
  sum_obj <- summary(x, ...)
  print.summary.fit.index(sum_obj)
  invisible(x)
}

#' @describeIn print Print method for \code{summary.fit.index} objects
#' @export
print.summary.fit.index <- function(x, ...) {

  digits <- x$digits

  printPackageInfo()
  cat("==============================================\n")
  cat("MODEL FIT INDICES\n")
  cat("==============================================\n")

  # Model Call
  cat("Call:\n")
  print(x$call)

  cat(sprintf("\nModel Information:\n"))
  cat(sprintf("  Sample Size (N): %d\n", x$data.info$N))
  cat(sprintf("  Free Parameters (npar): %s\n", x$fit.table$Value[x$fit.table$Statistic == "npar"]))

  cat("\nModel Fit Indices:\n")
  fit_data <- x$fit.table
  fit_data$Value_Formatted <- sprintf(paste0("%.", digits, "f"), as.numeric(fit_data$Value))

  print_table <- data.frame(
    Indices = fit_data$Statistic,
    Value = fit_data$Value_Formatted,
    Criterion = fit_data$Description,
    stringsAsFactors = FALSE
  )

  print_table <- print_table[print_table[["Indices"]] != "npar", , drop = FALSE]

  print(print_table, row.names = FALSE, print.gap = 2)

  cat("\n--- Note: Lower values are generally preferred for information criteria (AIC, BIC, etc.) ---\n")

  invisible(x)
}

#' @describeIn print Print method for \code{compare.model} objects
#' @export
print.compare.model <- function(x, ...) {
  sum_obj <- summary(x, ...)
  print.summary.compare.model(sum_obj)
  invisible(x)
}

#' @describeIn print Print method for \code{summary.compare.model} objects
#' @export
print.summary.compare.model <- function(x, ...) {
  digits <- x$digits

  printPackageInfo()
  cat("==============================================\n")
  cat("MODEL COMPARISON\n")
  cat("==============================================\n")

  cat("Call:\n")
  print(x$call)

  cat("\nData Information:\n")
  cat("  N =", x$data.info$N, "\n")
  cat("  I =", x$data.info$I, "\n")
  cat(paste0("  Classes compared: Model 1 (L=", x$data.info$L[1], ") vs Model 2 (L=", x$data.info$L[2], ")\n"))

  cat("\nModel Fit Statistics:\n")
  print(x$fit.table, row.names = FALSE)

  cat("\nClassification Performance:\n")
  colnames(x$model_comparison) <- c("Classes", "npar", "AvePP", "Entropy")
  print(x$model_comparison, row.names = FALSE)

  cat("\nBayes Factor (BF):", ifelse(is.na(x$BF), "Not available", format(x$BF, digits = x$digits)), "\n")
  cat("Interpretation:", x$BF_interpretation, "\n")

  if (!is.null(x$lrt_table)) {
    cat("\nLikelihood Ratio Tests:\n")
    names(x$lrt_table)[names(x$lrt_table) == "p-value"] <- "p.value"
    print(x$lrt_table, row.names = FALSE)
    cat("Signif. codes:  *** < 0.001, ** < 0.01, * < 0.05\n")
  } else {
    cat("\nNo Likelihood Ratio Tests available.\n")
  }

  cat("\n")
  invisible(x)
}

#' @describeIn print Print method for \code{SE} objects
#' @export
print.SE <- function(x, digits = 4, I.max = 5, L.max = 3, ...) {

  printPackageInfo()
  cat("==============================================\n")
  cat("STANDARD ERRORS\n")
  cat("==============================================\n")

  cat("\nCall:\n")
  print(x$call)

  model_type <- if (!is.null(x$se$means)) "LPA" else if (!is.null(x$se$par)) "LCA" else "Unknown"

  # Class probabilities SEs (always show all)
  cat("\nClass Probability Standard Errors:\n")
  print(round(x$se$P.Z, digits), digits = digits)
  cat("\n")

  # Model-specific parameters
  if (model_type == "LPA") {
    cat("Latent Profile Means Standard Errors:\n")

    # Determine display dimensions
    L_to_show <- min(L.max, nrow(x$se$means))
    I_to_show <- min(I.max, ncol(x$se$means))

    # Create truncated matrix with proper dimnames
    means_se <- round(x$se$means[1:L_to_show, 1:I_to_show, drop = FALSE], digits)
    rownames(means_se) <- paste0("Class", 1:L_to_show)
    colnames(means_se) <- colnames(x$se$means)[1:I_to_show]

    print(means_se, digits = digits)

    if (nrow(x$se$means) > L.max || ncol(x$se$means) > I.max) {
      cat(sprintf("\n[Truncated to first %d classes and %d variables; use $se$means to access full array]\n",
                  L_to_show, I_to_show))
    }

    # Optional: Show covariance SEs summary
    cat("\nCovariance Parameters Standard Errors:\n")
    cat("  Total non-zero SEs:", sum(x$se$covs != 0, na.rm = TRUE), "/", length(x$se$covs), "\n")
    cat("  [Use extract(x, 'covs') to access full array]\n")

  } else if (model_type == "LCA") {
    cat("Item Response Probability Standard Errors:\n")

    # Determine display dimensions
    L_to_show <- min(L.max, dim(x$se$par)[1])
    n_items_to_show <- min(I.max, dim(x$se$par)[2])
    n_cats <- dim(x$se$par)[3]

    # Show first item as example
    if (n_items_to_show > 0) {
      cat(sprintf("\nItem 1 (first %d classes, all categories):\n", L_to_show))

      # Extract and format first item's SEs
      item_se <- round(x$se$par[1:L_to_show, 1, , drop = FALSE], digits)
      dimnames(item_se) <- list(
        Class = paste0("Class", 1:L_to_show),
        Item = "Item1",
        Category = paste0("Cat", 1:n_cats)
      )

      # Print as matrix for readability
      item_se_mat <- matrix(item_se, nrow = L_to_show,
                            dimnames = list(paste0("Class", 1:L_to_show),
                                            paste0("Cat", 1:n_cats)))
      print(item_se_mat, digits = digits)

      cat("  [Use extract(x, 'par') to access full array]\n")
    }

    cat("\nTotal non-zero parameter SEs:", sum(x$se$par != 0, na.rm = TRUE), "/", length(x$se$par), "\n")
  }

  # Diagnostics section
  cat("\n")
  cat(strrep("-", 30), "\n")
  cat("Diagnostics:\n")
  if (x$diagnostics$method == "Bootstrap") {
    req <- x$diagnostics$n.Bootstrap.requested
    comp <- if (!is.null(x$diagnostics$n.Bootstrap.completed)) x$diagnostics$n.Bootstrap.completed else req
    cat(sprintf("  Bootstrap replicates: %d completed / %d requested (%.1f%%)\n",
                comp, req, 100 * comp/req))
    if (comp < req) cat("  ! Warning: Incomplete replicates may reduce accuracy\n")
  } else if (x$diagnostics$method == "Obs") {
    cond_num <- if (!is.null(x$diagnostics$hessian_cond_number)) x$diagnostics$hessian_cond_number else NA
    if (!is.na(cond_num)) {
      cat(sprintf("  Hessian condition number: %.2e\n", cond_num))
      if (cond_num > 1e6) {
        cat("  ! Warning: High condition number (>1e6) indicates unstable SE estimates\n")
      }
    }
  }

  invisible(x)
}

#' @describeIn print Print method for summary.SE objects
#' @export
print.summary.SE <- function(x, ...) {

  printPackageInfo()
  cat("==============================================\n")
  cat("STANDARD ERRORS\n")
  cat("==============================================\n")

  cat("Call:\n")
  print(x$call)

  # Model information
  cat("Model Type:", ifelse(x$model_type == "LPA", "Latent Profile Analysis", "Latent Class Analysis"), "\n")
  cat("Latent Classes:", x$L, "\n")
  if (!is.na(x$I)) {
    cat(ifelse(x$model_type == "LPA", "Variables", "Items"), ":", x$I, "\n")
  }

  # Parameter coverage
  cat("\nParameter Coverage:\n")
  cat(sprintf("  Class Probabilities: %d/%d non-zero SEs\n",
              x$nonzero_counts$P.Z, x$total_PZ))

  if (x$model_type == "LPA") {
    cat(sprintf("  Means Parameters   : %d/%d non-zero SEs\n",
                x$nonzero_counts$means, x$L * x$I))
    cat(sprintf("  Covariance Params  : %d non-zero SEs\n",
                x$nonzero_counts$covs))
  } else if (x$model_type == "LCA") {
    total_params <- sum(sapply(1:x$I, function(i) {
      # Assuming each item has varying categories - this is approximate
      x$L * (ifelse(i <= length(x$nonzero_counts), x$nonzero_counts[[i]], 2) - 1)
    }))
    cat(sprintf("  Item Parameters    : %d non-zero SEs\n",
                x$nonzero_counts$par))
  }

  # Diagnostics
  cat("\nDiagnostics:\n")
  if (x$method == "Bootstrap") {
    req <- x$diagnostics$n.Bootstrap.requested
    comp <- if (!is.null(x$diagnostics$n.Bootstrap.completed)) x$diagnostics$n.Bootstrap.completed else req
    cat(sprintf("  Bootstrap replicates: %d completed / %d requested (%.1f%%)\n",
                comp, req, 100 * comp/req))
  } else if (x$method == "Obs") {
    cond_num <- if (!is.null(x$diagnostics$hessian_cond_number)) x$diagnostics$hessian_cond_number else NA
    if (!is.na(cond_num)) {
      cat(sprintf("  Hessian condition number: %.2e\n", cond_num))
    }
  }

  cat("\nCall: ")
  print(x$call)
  cat("\n")

  invisible(x)
}
