% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pick_kols.R
\name{pick_kols}
\alias{pick_kols}
\title{Pick key opinion leaders from a network given constraints}
\usage{
pick_kols(
  network,
  tosource = TRUE,
  goal = "diffusion",
  m = 1,
  range = c(1, 1),
  top = NULL,
  include = NULL,
  exclude = NULL,
  attribute = NULL,
  alpha = 0.9,
  beta = 0.9,
  file = NULL
)
}
\arguments{
\item{network}{a unipartite unweighted network as an adjacency \code{matrix} or \code{igraph} object}

\item{tosource}{logical: edges point \emph{toward} a source of information}

\item{goal}{string: goal for the KOL team (either \code{"diffusion"} or \code{"adoption"})}

\item{m}{integer: KOL team centrality parameter (\code{m == 1} is equivalent to simple degree centrality)}

\item{range}{vector: a vector of length 2 containing the minimum and maximum number of KOLs on a team}

\item{top}{numeric: restrict scope to the \code{top} nodes with the highest degree, closeness, or betweenness (useful for large networks)}

\item{include}{vector: names or indices of nodes that \strong{must} be included on the KOL team}

\item{exclude}{vector: names or indices of nodes that \strong{can not} be included on the KOL team}

\item{attribute}{string or vector: if \code{network} is an \code{igraph} object, the name of a node attribute. if \code{network} is an adjacency matrix, a vector containing a node attribute.}

\item{alpha}{numeric: parameter to control relative weight of breadth and diversity in overall evaluation of KOL teams (\eqn{0.5 \leq \alpha \leq 1})}

\item{beta}{numeric: parameter to control weight of team size in overall evaluation of KOL teams (\eqn{0 \leq \beta \leq 2})}

\item{file}{string: filename to write a sorted list of possible KOL teams as a CSV.}
}
\value{
A sorted list containing a data frame of possible KOL teams with their characteristics, the \code{network}, \code{m}, \code{goal}, and (optionally) \code{attribute}
}
\description{
Pick key opinion leaders from a network given constraints
}
\details{
When seeking to diffuse a piece of information or encourage adoption of a behavior, it is often useful
to recruit the assistance of \emph{key opinion leaders} (KOL) in a network. \code{pick_kols} facilitates selecting
members of a KOL team by returning a dataframe of possible teams. The selection of a KOL team often depends on
several factors, which this function summarizes as ABCDE:
\itemize{
\item Availability - The availability of individuals to serve as a KOL. This can be controlled by the \code{include} and \code{exclude} parameters.
\item Breadth - The fraction of non-KOLs that the KOL team can influence. When \code{goal=="diffusion"}, breadth is measured as the fraction of non-KOLs that a KOL team can reach in \code{m} steps (i.e., m-reach). When \code{goal=="adoption"}, breadth is measured as the fraction of non-KOLs that are directly connected to at least \code{m} KOLs (i.e., m-contact).
\item Cost - The number of KOLs to be recruited and trained (i.e., team size).
\item Diversity - The fraction of values of \code{attribute} represented on the KOL team.
\item Evaluation - Potential KOL teams must be compared and evaluated in a way that balances these considerations.
}

\strong{Evaluating KOL Teams}

Potential KOL teams are evaluated on the basis of breadth (B), Cost (C), and (if \code{attribute} is provided), Diversity (D)
using \deqn{\frac{B}{C^\beta} \mbox{   or   } \frac{B^\alpha D^{1-\alpha}}{C^\beta}}

The \eqn{\alpha} parameter can take values \eqn{0.5 < \alpha < 1} and controls the weight placed on breadth relative to diversity.
Smaller values of \eqn{\alpha} place less weight on breadth and more weight on diversity, while larger values of \eqn{\alpha}
place more weight on breadth and less weight on diversity. The default (\eqn{\alpha = 0.9}) places the majority of weight on
the breadth of the network that KOL teams cover, while still considering the team's diversity (primarily as a tie-breaker).

The \eqn{\beta} parameter can take values \eqn{0 < \beta < 2} and controls the cost of larger KOL team members. Smaller values of
\eqn{\beta} imply decreasing marginal costs, while larger values of \eqn{\beta} imply increasing marginal costs. The default
(\eqn{\beta = 0.9}) assumes that team members have a slight diminishing marginal cost (i.e. the cost of each additional
team member is slightly smaller than the previous one).

\strong{Interpreting Edge Direction}

If \code{network} is a directed network, then \code{tosource} controls how the direction of edges is interpreted:
\itemize{
\item \code{tosource = TRUE} (default) - An edge i -> j is interpreted as "i gets information from j" or "i is influenced
by j" (i.e., the edge points \emph{toward} a source of information or influence). This type of data usually results from
asking respondents to nominate the people from whom they seek advice. In this case, actors with high \emph{in-degree} like
\emph{j} are generally better KOLs.
\item \code{tosource = FALSE} - An edge i -> j is interpreted as "i sends information to j" or "i influences j" (i.e., the
edge points \emph{away} from a source of information or influence). This type of data usually results from asking respondents
to report the people to whom they give advice. In this case, actors with high \emph{out-degree} like \emph{i} are generally better KOLs.
}
}
\examples{
network <- igraph::sample_smallworld(1,26,2,.2)  #An example network
igraph::V(network)$name <- letters[1:26]  #Give the nodes names
igraph::V(network)$gender <- sample(c("M","F"),26,replace=TRUE)  #Give the nodes a "gender"
teams <- pick_kols(network,              #Find KOL teams in `network`
                   m = 2,
                   range = c(2,4),       #containing 2-4 members
                   attribute = "gender", #that are gender diverse
                   goal = "diffusion")   #and can help diffuse information
teams$teams[1:10,]  #Look at the top 10 teams
}
