% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/accessors.R
\name{get_CLAN}
\alias{get_CLAN}
\title{Accessor function for the CLAN generic target estimates}
\usage{
get_CLAN(x, variable, learner = "best", plot = TRUE)
}
\arguments{
\item{x}{An object of the class \code{"\link{GenericML}"}, as returned by the function \code{\link{GenericML}()}.}

\item{variable}{The (character) name of a variabe on which CLAN was performed.}

\item{learner}{A character string of the learner whose CLAN generic target estimates shall be accessed. Default is \code{"best"} for the best learner for CLAN}

\item{plot}{Logical. If \code{TRUE} (default), a \code{"\link[ggplot2]{ggplot}"} object is computed.}
}
\value{
An object of class \code{"CLAN_info"}, which consists of the following components:
\describe{
\item{\code{estimate}}{A numeric vector of point estimates of the CLAN generic targets.}
\item{\code{confidence_interval}}{A numeric matrix of the lower and upper confidence bounds for each generic target. The confidence level of the implied confidence interval is equal to \code{1 - 2 * significance_level}.}
\item{\code{confidence_level}}{The confidence level of the confidence intervals. Equals \code{1 - 2 * significance_level}.}
\item{\code{learner}}{The argument \code{learner}.}
\item{\code{plot}}{An object of class \code{"\link[ggplot2]{ggplot}"}. Only returned if the argument \code{plot = TRUE}.}
\item{\code{CLAN_variable}}{The name of the CLAN variable of interest.}
}
}
\description{
Accessor function for the CLAN generic target estimates
}
\examples{
if(require("rpart") && require("ranger")){
## generate data
set.seed(1)
n  <- 150                                  # number of observations
p  <- 5                                    # number of covariates
D  <- rbinom(n, 1, 0.5)                    # random treatment assignment
Z  <- matrix(runif(n*p), n, p)             # design matrix
Y0 <- as.numeric(Z \%*\% rexp(p) + rnorm(n)) # potential outcome without treatment
Y1 <- 2 + Y0                               # potential outcome under treatment
Y  <- ifelse(D == 1, Y1, Y0)               # observed outcome

## column names of Z
colnames(Z) <- paste0("V", 1:p)

## specify learners
learners <- c("tree", "mlr3::lrn('ranger', num.trees = 10)")

## perform generic ML inference
# small number of splits to keep computation time low
x <- GenericML(Z, D, Y, learners, num_splits = 2,
               parallel = FALSE)

## access best learner
get_best(x)

## access BLP generic targets for best learner w/o plot
get_BLP(x, learner = "best", plot = FALSE)

## access BLP generic targets for ranger learner w/o plot
get_BLP(x, learner = "mlr3::lrn('ranger', num.trees = 10)", plot = FALSE)

## access GATES generic targets for best learner w/o plot
get_GATES(x, learner = "best", plot = FALSE)

## access GATES generic targets for ranger learner w/o plot
get_GATES(x, learner = "mlr3::lrn('ranger', num.trees = 10)", plot = FALSE)

## access CLAN generic targets for "V1" & best learner, w/o plot
get_CLAN(x, learner = "best", variable = "V1", plot = FALSE)

## access CLAN generic targets for "V1" & ranger learner, w/o plot
get_CLAN(x, learner = "mlr3::lrn('ranger', num.trees = 10)",
         variable = "V1", plot = FALSE)
}

}
\seealso{
\code{\link{GenericML}()},
\code{\link{get_BLP}()},
\code{\link{get_GATES}()},
\code{\link{get_best}()},
\code{\link{print.BLP_info}()},
\code{\link{print.GATES_info}()},
\code{\link{print.CLAN_info}()}
}
