
#' @export
#' @title An Example Illustrating OLG Stochastic Spot Equilibrium Paths
#' @aliases gemOLG_StochasticSpotEquilibrium_3_3
#' @description
#' An example illustrating OLG stochastic spot equilibrium paths.
#' The economy includes consumers with a lifespan of two periods and a type of spot-trading firm (i.e., a spot-trading enterprise).
#' The economy features three types of commodities: products, labor, and public bonds, with the latter serving as a means of storing value.
#'
#' Consumers in the economy live for two periods: youth and old age.
#' Consumers only consume the product.
#'
#' Young people save by purchasing public bonds.
#' The savings rate of young consumers is an exogenous variable.
#' This savings rate determines how each young person's income is distributed between their youth and old age.
#'
#' Young people supply a total of 100 units of labor each period;
#' old people supply a total of 100 units of public bonds each period.
#'
#' The spot-trading enterprise operates independent of consumers,
#' requiring products and labor for production, with a production function of \eqn{\alpha\sqrt{x_1 x_2}}.
#' Assume \eqn{\alpha} follows a first-order autoregressive process.
#' The enterprise's objective is to maximize output, without making forecasts for the future.
#' @param ... arguments to be passed to the function sdm2.
#' @note
#' The savings rate of the youth in the program is set to 0.5, meaning the savings-to-consumption ratio is 1.
#'
#' Public bonds are used as the numeraire.
#'
#' The initial output (i.e., initial product supply) is set to 20 in the parameter z0.
#' @examples
#' \donttest{
#' set.seed(1)
#' dst.firm <- node_new(
#'   "prod",
#'   type = "CD", alpha = 1,
#'   beta = c(0.5, 0.5),
#'   "prod", "lab"
#' )
#'
#' dst.age1 <- node_new(
#'   "util",
#'   type = "FIN",
#'   rate = c(1, ratio.saving.consumption = 1),
#'   "prod", "publicBond"
#' )
#'
#' dst.age2 <- node_new(
#'   "util",
#'   type = "Leontief", a = 1,
#'   "prod"
#' )
#'
#' policy.technology <- function(time, A) {
#'   A[[1]]$alpha <- exp(0.95 * log(A[[1]]$alpha) +
#'                         rnorm(1, sd = 0.01))
#' }
#'
#' set.seed(1)
#' ge <- sdm2(
#'   A = list(
#'     dst.firm, dst.age1, dst.age2
#'   ),
#'   B = matrix(c(
#'     1, 0, 0,
#'     0, 0, 0,
#'     0, 0, 0
#'   ), 3, 3, TRUE),
#'   S0Exg = matrix(c(
#'     NA, NA, NA,
#'     NA, 100, NA,
#'     NA, NA, 100
#'   ), 3, 3, TRUE),
#'   names.commodity = c("prod", "lab", "publicBond"),
#'   names.agent = c("firm", "age1", "age2"),
#'   numeraire = "publicBond",
#'   maxIteration = 1,
#'   numberOfPeriods = 20,
#'   policy = list(
#'     policy.technology,
#'     policyMarketClearingPrice
#'   ),
#'   z0 = c(20, 1, 1),
#'   ts = TRUE
#' )
#'
#' matplot(ge$ts.z, type = "o", pch = 20)
#' }

gemOLG_StochasticSpotEquilibrium_3_3 <- function(...) sdm2(...)
