#' Compare distributions of numerical variables between subsets
#'
#' Sets up and calls WelchRankTest to compare the distributions of
#' a set of numerical variables between two record subsets.  If the
#' set of numerical variables contains a single element, this function
#' effectively reduces to WelchRankTest.
#'
#' @param DF data frame containing all variables in `numVars`
#' @param IndexA record index defining the first record subset to be compared
#' @param numVars vector of numerical variable names from `DF`
#' @param IndexB record index defining the second record subset to be compared
#' (default NULL means the second set contains all records not included in the first)
#' @param cLevel confidence level for the Welch rank test (default = 0.95)
#'
#' @return  data frame with one row for each element of `numVars` and columns
#' containing the `numVars` element name and all columns from WelchRankTest
#' for that variable
#' @export
#'
#' @examples
#' x <- seq(-1, 1, length = 200)
#' a <- rep(c("a", "b"), 100)
#' offset <- rep(c(0, 0.2), 100)
#' xMod <- x + offset
#' DF <- data.frame(numVar = x, numVar2 = xMod, setVar = a)
#' indexA <- which(DF$setVar == "a")
#' CompareNumericSets(DF, indexA, c("numVar", "numVar2"))
CompareNumericSets <- function(DF, IndexA, numVars, IndexB = NULL, cLevel = 0.95){
  #
  stopifnot("DF must be a data frame"= is.data.frame(DF))
  #
  if (is.null(IndexB)){
    IndexB <- setdiff(seq(1, nrow(DF), 1), IndexA)
  }
  outFrame <- NULL
  n <- length(numVars)
  stopifnot("numVars list is empty"= n > 0)
  for (i in 1:n){
    upFrame <- WelchRankTest(DF, numVars[i], IndexA, IndexB, cLevel)
    outFrame <- rbind.data.frame(outFrame, upFrame)
  }
  names(outFrame) <- names(upFrame)
  outFrame$Variable <- numVars
  #
  allVars <- colnames(outFrame)
  varIndex <- which(allVars == "Variable")
  newIndex <- c(varIndex, setdiff(1:ncol(outFrame), varIndex))
  outFrame <- outFrame[, newIndex]
  #
  return(outFrame)
}


