% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EllDistrEst-adaptive.R
\name{estim_tilde_AMSE}
\alias{estim_tilde_AMSE}
\title{Estimate the part of the AMSE of the elliptical density generator that only depends
on the parameter "a" assuming \eqn{h} has been optimally chosen}
\usage{
estim_tilde_AMSE(
  X,
  mu = 0,
  Sigma_m1 = diag(NCOL(X)),
  grid,
  h,
  Kernel = "gaussian",
  a = 1,
  mpfr = FALSE,
  precBits = 100,
  dopb = TRUE
)
}
\arguments{
\item{X}{a matrix of size \eqn{n \times d}, assumed to be \eqn{n} i.i.d. observations
(rows) of a \eqn{d}-dimensional elliptical distribution.}

\item{mu}{mean of X. This can be the true value or an estimate. It must be
a vector of dimension \eqn{d}.}

\item{Sigma_m1}{inverse of the covariance matrix of X.
This can be the true value or an estimate. It must be
a matrix of dimension \eqn{d \times d}.}

\item{grid}{grid of values of \eqn{\xi} at which we want to estimate the
density generator.}

\item{h}{bandwidth of the kernel. Can be either a number or a vector of the
size \code{length(grid)}.}

\item{Kernel}{name of the kernel. Possible choices are
\code{"gaussian"}, \code{"epanechnikov"}, \code{"triangular"}.}

\item{a}{tuning parameter to improve the performance at 0.
Can be either a number or a vector of the
size \code{length(grid)}. If this is a vector, the code will need to allocate
a matrix of size \code{nrow(X) * length(grid)} which can be prohibitive in
some cases.}

\item{mpfr}{if \code{mpfr = TRUE}, multiple precision floating point is used
via the package \link[Rmpfr:Rmpfr]{Rmpfr}.
This allows for a higher (numerical) accuracy, at the expense of computing time.
It is recommended to use this option for higher dimensions.}

\item{precBits}{number of precBits used for floating point precision
(only used if \code{mpfr = TRUE}).}

\item{dopb}{a Boolean value.
If \code{dopb = TRUE}, a progress bar is displayed.}
}
\value{
a vector of the same size as the grid, with the corresponding value
for the \eqn{\widetilde{AMSE}}.
}
\description{
A continuous elliptical distribution has a density of the form
\deqn{f_X(x) = {|\Sigma|}^{-1/2}
g\left( (x-\mu)^\top \, \Sigma^{-1} \, (x-\mu) \right),
}
where \eqn{x \in \mathbb{R}^d},
\eqn{\mu \in \mathbb{R}^d} is the mean,
\eqn{\Sigma} is a \eqn{d \times d} positive-definite matrix
and a function \eqn{g: \mathbb{R}_+ \rightarrow \mathbb{R}_+}, called the
density generator of \eqn{X}.
The goal is to estimate \eqn{g} at some point \eqn{\xi}, by
\deqn{
\widehat{g}_{n,h,a}(\xi)
:= \dfrac{\xi^{\frac{-d+2}{2}} \psi_a'(\xi)}{n h s_d}
\sum_{i=1}^n
  K\left( \dfrac{ \psi_a(\xi) - \psi_a(\xi_i) }{h} \right)
+ K\left( \dfrac{ \psi_a(\xi) + \psi_a(\xi_i) }{h} \right),
}
where
\eqn{s_d := \pi^{d/2} / \Gamma(d/2)},
\eqn{\Gamma} is the Gamma function,
\eqn{h} and \eqn{a} are tuning parameters (respectively the bandwidth and a
parameter controlling the bias at \eqn{\xi = 0}),
\eqn{\psi_a(\xi) := -a + (a^{d/2} + \xi^{d/2})^{2/d},}
\eqn{\xi \in \mathbb{R}}, \eqn{K} is a kernel function and
\eqn{\xi_i := (X_i - \mu)^\top \, \Sigma^{-1} \, (X_i - \mu),
}
for a sample \eqn{X_1, \dots, X_n}.
Thanks to Proposition 2.2 in (Ryan and Derumigny, 2024), the asymptotic
mean square error of \eqn{\widehat{g}_{n,h,a}(\xi)} can be decomposed into
a product of a constant (that depends on the true \eqn{g}) and a term that
depends on \eqn{g} and \eqn{a}. This function computes this term. It can be
useful to find out the best value of the parameter \eqn{a} to be used.
}
\examples{
# Comparison between the estimated and true generator of the Gaussian distribution
n = 50000
d = 3
X = matrix(rnorm(n * d), ncol = d)
grid = seq(0, 5, by = 0.1)
a = 1.5

AMSE_est = estim_tilde_AMSE(X = X, grid = grid, a = a, h = 0.09)
plot(grid, abs(AMSE_est), type = "l")

# Computation of true values
g = exp(-grid/2)/(2*pi)^{3/2}
gprime = (-1/2) *exp(-grid/2)/(2*pi)^{3/2}
A = a^(d/2)
psia = -a + (A + grid^(d/2))^(2/d)
psiaprime = grid^(d/2 - 1) * (A + grid^(d/2))^(2/d - 1)
psiasecond = psiaprime * ( (d-2)/2 ) * grid^{-1} * A *
  ( grid^(d/2) + A )^(-1)

rhoprimexi = ((d-2) * grid^((d-4)/2) * psiaprime
  - 2 * grid^((d-2)/2) * psiasecond) / (2 * psiaprime^3) * g +
  grid^((d-2)/2) / (psiaprime^2) * gprime

AMSE = rhoprimexi / psiaprime

lines(grid, abs(AMSE), col = "red")


# Comparison as a function of $a$
n = 50000
d = 3
X = matrix(rnorm(n * d), ncol = d)
grid = 0.1
vec_a = c(0.001, 0.002, 0.005,
0.01, 0.02, 0.05, 0.1, 0.2, 0.3, 0.4, 0.5, 0.8, 1, 1.5, 2)

AMSE_est = rep(NA, length = length(vec_a))
for (i in 1:length(vec_a)){
  AMSE_est[i] = estim_tilde_AMSE(X = X, grid = grid, a = vec_a[i], h = 0.09,
                          dopb = FALSE)
}

plot(vec_a, abs(AMSE_est), type = "l", log = "x")

# Computation of true values
a = vec_a

g = exp(-grid/2)/(2*pi)^{3/2}
gprime = (-1/2) *exp(-grid/2)/(2*pi)^{3/2}
A = a^(d/2)
psia = -a + (A + grid^(d/2))^(2/d)
psiaprime = grid^(d/2 - 1) * (A + grid^(d/2))^(2/d - 1)
psiasecond = psiaprime * ( (d-2)/2 ) * grid^{-1} * A *
  ( grid^(d/2) + A )^(-1)

rhoprimexi = ((d-2) * grid^((d-4)/2) * psiaprime
  - 2 * grid^((d-2)/2) * psiasecond) / (2 * psiaprime^3) * g +
  grid^((d-2)/2) / (psiaprime^2) * gprime

AMSE = rhoprimexi / psiaprime

yliminf = min(c(abs(AMSE_est), abs(AMSE)))
ylimsup = max(c(abs(AMSE_est), abs(AMSE)))

plot(vec_a, abs(AMSE_est), type = "l", log = "xy",
     ylim = c(yliminf, ylimsup))
lines(vec_a, abs(AMSE), col = "red")

}
\references{
Ryan, V., & Derumigny, A. (2024).
On the choice of the two tuning parameters for nonparametric estimation of an
elliptical distribution generator
\href{https://arxiv.org/abs/2408.17087}{arxiv:2408.17087}.
}
\author{
Alexis Derumigny, Victor Ryan
}
