% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EllDistrEstDeriv.R
\name{EllDistrDerivEst}
\alias{EllDistrDerivEst}
\title{Estimate the derivatives of a generator}
\usage{
EllDistrDerivEst(
  X,
  mu = 0,
  Sigma_m1 = diag(NCOL(X)),
  grid,
  h,
  Kernel = "gaussian",
  a = 1,
  k,
  mpfr = FALSE,
  precBits = 100,
  dopb = TRUE
)
}
\arguments{
\item{X}{a matrix of size \eqn{n \times d}, assumed to be \eqn{n} i.i.d. observations
(rows) of a \eqn{d}-dimensional elliptical distribution.}

\item{mu}{mean of X. This can be the true value or an estimate. It must be
a vector of dimension \eqn{d}.}

\item{Sigma_m1}{inverse of the covariance matrix of X.
This can be the true value or an estimate. It must be
a matrix of dimension \eqn{d \times d}.}

\item{grid}{grid of values on which to estimate the density generator.}

\item{h}{bandwidth of the kernel. Can be either a number or a vector of the
size \code{length(grid)}.}

\item{Kernel}{name of the kernel. Possible choices are
\code{"gaussian"}, \code{"epanechnikov"}, \code{"triangular"}.}

\item{a}{tuning parameter to improve the performance at 0.}

\item{k}{highest order of the derivative of the generator that is to be
estimated. For example, \code{k = 1} corresponds to the estimation of the
generator and of its derivative. \code{k = 2} corresponds to the estimation
of the generator as well as its first and second derivatives.}

\item{mpfr}{if \code{mpfr = TRUE}, multiple precision floating point is used
via the package \link[Rmpfr:Rmpfr]{Rmpfr}.
This allows for a higher (numerical) accuracy, at the expense of computing time.
It is recommended to use this option for higher dimensions.}

\item{precBits}{number of precBits used for floating point precision
(only used if \code{mpfr = TRUE}).}

\item{dopb}{a Boolean value.
If \code{dopb = TRUE}, a progress bar is displayed.}
}
\value{
a matrix of size \code{length(grid) * (kmax + 1)}
with the estimated value of the generator and all its derivatives
at all orders until and including \code{kmax}, at all points of the grid.
}
\description{
A continuous elliptical distribution has a density of the form
\deqn{f_X(x) = {|\Sigma|}^{-1/2}
g\left( (x-\mu)^\top \, \Sigma^{-1} \, (x-\mu) \right),
}
where \eqn{x \in \mathbb{R}^d},
\eqn{\mu \in \mathbb{R}^d} is the mean,
\eqn{\Sigma} is a \eqn{d \times d} positive-definite matrix
and a function \eqn{g: \mathbb{R}_+ \rightarrow \mathbb{R}_+}, called the
density generator of \eqn{X}.
The goal is to estimate the derivatives of \eqn{g} at some point \eqn{\xi},
by kernel smoothing, following Section 3 of (Ryan and Derumigny, 2024).
}
\details{
Note that this function may be rather slow for higher-order derivatives.
Furthermore, it is likely that the number of observations needs to be quite
high for the higher-order derivatives to be estimated well enough.
}
\examples{

# Comparison between the estimated and true generator of the Gaussian distribution
n = 50000
d = 3
X = matrix(rnorm(n * d), ncol = d)
grid = seq(0, 5, by = 0.1)
a = 1.5

gprimeEst = EllDistrDerivEst(X = X, grid = grid, a = a, h = 0.09, k = 1)[,2]
plot(grid, gprimeEst, type = "l")

# Computation of true values
g = exp(-grid/2)/(2*pi)^{3/2}
gprime = (-1/2) * exp(-grid/2)/(2*pi)^{3/2}

lines(grid, gprime, col = "red")

}
\references{
Ryan, V., & Derumigny, A. (2024).
On the choice of the two tuning parameters for nonparametric estimation of an
elliptical distribution generator
\href{https://arxiv.org/abs/2408.17087}{arxiv:2408.17087}.
}
\seealso{
\code{\link{EllDistrEst}} for the nonparametric estimation of the
elliptical distribution density generator itself,
\code{\link{EllDistrSim}} for the simulation of elliptical distribution samples.

This function uses the internal functions \code{\link{compute_etahat}}
and \code{\link{compute_matrix_alpha}}.
}
\author{
Alexis Derumigny, Victor Ryan

Victor Ryan, Alexis Derumigny
}
