#' Scree Test Optimal Coordinate (STOC)
#'
#' @description
#' The STOC (Raiche et al., 2013) approach is based on fitting \eqn{I-2} (\eqn{I}
#' is the number of observed variables) two
#' point regression models using the (i+1)th and I-th (the last one) eigenvalue to
#' obtain a prediction for the i-th eigenvalue that is referred to as optimal
#' coordinate (with p being the number of observed variables and therefore
#' also the number of eigenvalues).  If the observed empirical eigenvalue is larger than the predicted
#' optimal coordinate. Raiche et al. (2013) also argue for a restricted version that retains
#' only factors with an eigenvalue greater than the reference eigenvalue by PA (holds PCA and
#' quant=0.95 in this package).
#' @seealso \code{\link[EFAfactors]{STOC}}
#'
#' @param response A required \code{N} × \code{I} matrix or data.frame consisting of the responses
#'        of \code{N} individuals to \code{I} items.
#' @param fa A string that determines the method used to obtain eigenvalues in PA. If \code{"pc"}, it represents
#'           Principal Component Analysis (PCA); if \code{"fa"}, it represents Principal Axis Factoring (a widely
#'           used Factor Analysis method; @seealso \code{\link[EFAfactors]{factor.analysis}};
#'           Auerswald & Moshagen, 2019). (Default = \code{"pc"})
#' @param nfact A numeric value that specifies the number of factors to extract, only effective when \code{fa = 'fa'}. (Default = 1)
#' @param cor.type A character string indicating which correlation coefficient (or covariance) is
#'                to be computed. One of \code{"pearson"} (default), \code{"kendall"}, or
#'                \code{"spearman"}. @seealso \code{\link[stats]{cor}}.
#' @param use an optional character string giving a method for computing covariances in the presence of missing values. This
#'          must be one of the strings \code{"everything"}, \code{"all.obs"}, \code{"complete.obs"}, \code{"na.or.complete"},
#'          or \code{"pairwise.complete.obs"} (default). @seealso \code{\link[stats]{cor}}.
#' @param vis A Boolean variable that will print the factor retention results when set to \code{TRUE}, and will not print
#'          when set to \code{FALSE}. (default = \code{TRUE})
#' @param plot A Boolean variable that will print the STOC plot when set to \code{TRUE}, and will not print it when set to
#'          \code{FALSE}. @seealso \code{\link[EFAfactors]{plot.STOC}}. (Default = \code{TRUE})
#'
#' @return An object of class \code{STOC}, which is a \code{list} containing the following components:
#' \item{nfact}{The number of factors to retain by both Optimal Coordinate and PA.}
#' \item{nfact.STOC}{The number of factors to retain only by Optimal Coordinate.}
#' \item{nfact.PA}{The number of factors to retain only by PA.}
#' \item{fa}{Indicates the method used to obtain eigenvalues. 'pc' represents Principal Component Analysis, and 'fa' represents Principal Axis Factoring.}
#' \item{eigen.value}{A vector containing the empirical eigenvalues.}
#' \item{eigen.ref}{A vector containing the reference eigenvalues by both Optimal Coordinate and PA.}
#'
#' @references
#'  Raiche, G., Walls, T. A., Magis, D., Riopel, M., & Blais, J.-G. (2013). Non-graphical solutions for Cattell's scree test. Methodology, 9(1), 23-29. https://doi.org/10.1027/1614-2241/a000051
#'
#'  Goretzko, D. (2025). How many factors to retain in exploratory factor analysis? A critical overview of factor retention methods. Psychological methods, Advance online publication. https://doi.org/10.1037/met0000733
#'
#' @examples
#' library(EFAfactors)
#' set.seed(123)
#'
#' ##Take the data.bfi dataset as an example.
#' data(data.bfi)
#'
#' response <- as.matrix(data.bfi[, 1:25]) ## loading data
#' response <- na.omit(response) ## Remove samples with NA/missing values
#'
#' ## Transform the scores of reverse-scored items to normal scoring
#' response[, c(1, 9, 10, 11, 12, 22, 25)] <- 6 - response[, c(1, 9, 10, 11, 12, 22, 25)] + 1
#'
#' \donttest{
#'  STOC.obj <- STOC(response, plot=FALSE)
#'
#'  ## STOC plot
#'  plot(STOC.obj)
#'
#' }
#'
#' @export
STOC <- function (response, fa = "pc", nfact = 1,
                  cor.type = "pearson", use = "pairwise.complete.obs",
                  vis = TRUE, plot = TRUE) {
  if (!any(rep(use, 5) == c("everything", "all.obs", "complete.obs",
                            "na.or.complete", "pairwise.complete.obs")))
    stop("'use' must be one of the strings 'everything', 'all.obs', 'complete.obs', 'na.or.complete', or 'pairwise.complete.obs' !")
  if (!any(rep(fa, 2) == c("pc", "fa")))
    stop("'type' must be one of the strings 'fa' or 'pc' !")

  N <- dim(response)[1]
  I <- dim(response)[2]
  response <- scale(response)

  PA.obj <- PA(response, vis = FALSE, plot = FALSE)

  if (fa == "pc") {
    cor.response <- cor(response, method = cor.type, use = use)
    eigen.result <- eigen(cor.response)
    eigen.value <- eigen.result$values
  }else {
    eigen.value <- as.vector(factor.analysis(response, nfact = nfact,
                                             cor.type = cor.type, use = use)$eigen.value)
  }

  vp <- eigen.value[I]
  eigen.ref <- rep(0, I)
  for(i in 1:(I-1)){
    b.i_1 <- (vp - eigen.value[i+1]) / (I-i-1)
    a.i_1 <- eigen.value[i+1] - b.i_1 * (i+1)

    lambda.value <- i

    eigen.ref[i] <- a.i_1 + lambda.value*b.i_1
  }

  nfact <- 1
  while (eigen.value[nfact] > eigen.ref[nfact] & nfact < (I-2)) {
    nfact <- nfact + 1
  }
  nfact.STOC <- nfact - 1

  nfact <- min(nfact.STOC, PA.obj$nfact)

  STOC.obj <- list(eigen.value = eigen.value, eigen.ref=eigen.ref,
                   nfact=nfact,
                   nfact.STOC=nfact.STOC, nfact.PA=PA.obj$nfact,
                   fa=fa)
  class(STOC.obj) <- "STOC"

  if (vis)
    print(STOC.obj)
  if (plot)
    plot(STOC.obj)

  return(STOC.obj)
}
